//
//  Android PDF Writer
//  http://coderesearchlabs.com/androidpdfwriter
//
//  by Javier Santo Domingo (j-a-s-d@coderesearchlabs.com)
//

package crl.android.pdfwriter;

import android.graphics.Matrix;
import android.graphics.RectF;
import android.graphics.Rect;

import java.io.IOException;
import java.io.OutputStream;

public class PDFWriter {

    private PDFDocument mDocument;
    private IndirectObject mCatalog;
    private Pages mPages;
    private Page mCurrentPage;

    public PDFWriter() {
        newDocument(PaperSize.A4_PRINT_WIDTH, PaperSize.A4_PRINT_HEIGHT);
    }

    public PDFWriter(int pageWidth, int pageHeight) {
        newDocument(pageWidth, pageHeight);
    }

    /**
     * Resets existing document parameters and creates new document with the first initialized page
     *
     * @param pageWidth
     * @param pageHeight
     */
    public void newDocument(int pageWidth, int pageHeight) {
        mDocument = new PDFDocument();
        mCatalog = mDocument.newIndirectObject();
        mDocument.includeIndirectObject(mCatalog);
        mPages = new Pages(mDocument, pageWidth, pageHeight);
        mDocument.includeIndirectObject(mPages.getIndirectObject());
        renderCatalog();
    }

    public void setDocumentInfo(String info) {
        mDocument.setDocumentInfo(info);
    }

    private void renderCatalog() {
        mCatalog.setDictionaryContent("  /Type /Catalog\n  /Pages " + mPages.getIndirectObject().getIndirectReference() + "\n");
    }

    public void newPage(int pageWidth, int pageHeight) {
        mCurrentPage = mPages.newPage(pageWidth, pageHeight);
        mDocument.includeIndirectObject(mCurrentPage.getIndirectObject());
        mPages.render();
    }

    public void setCurrentPage(int pageNumber) {
        mCurrentPage = mPages.getPageAt(pageNumber);
    }

    public int getPageCount() {
        return mPages.getCount();
    }

    public void setFont(String subType, String baseFont) {
        mCurrentPage.setFont(subType, baseFont);
    }

    public void setFont(String subType, String baseFont, String encoding) {
        mCurrentPage.setFont(subType, baseFont, encoding);
    }

    public void addRawContent(String rawContent) {
        mCurrentPage.addRawContent(rawContent);
    }

    public void addText(int leftPosition, int topPositionFromBottom, int fontSize, String text) {
        addText(leftPosition, topPositionFromBottom, fontSize, text, Transformation.DEGREES_0_ROTATION);
    }

    public void addText(int leftPosition, int topPositionFromBottom, int fontSize, String text, String transformation) {
        mCurrentPage.addText(leftPosition, topPositionFromBottom, fontSize, text, transformation);
    }

    public void addLine(int fromLeft, int fromBottom, int toLeft, int toBottom) {
        mCurrentPage.addLine(fromLeft, fromBottom, toLeft, toBottom);
    }

    public void addRectangle(int fromLeft, int fromBottom, int toLeft, int toBottom) {
        mCurrentPage.addRectangle(fromLeft, fromBottom, toLeft, toBottom);
    }

    /**
     * Adds JPG image to the actual page
     */
    public void addJPGImageKeepRatio(int pageWidth, int pageHeight, int rotationDegrees, String path) throws IOException {
        JPGXObjectImage img = new JPGXObjectImage(mDocument, path);
        float imageWidth = img.getWidth();
        float imageHeight = img.getHeight();
        final float imgRatio = imageWidth / imageHeight;

        Matrix boxMatrix = new Matrix();
        boxMatrix.setRotate(rotationDegrees);
        RectF box = new RectF(0f, 0f, pageWidth, pageHeight);
        boxMatrix.mapRect(box);
        float transformedPageWidth = (int) box.width();
        float transformedPageHeight = (int) box.height();

        final float boxRatio = transformedPageWidth / transformedPageHeight;

        float ratio;
        if (imgRatio > boxRatio) {
            ratio = transformedPageWidth / imageWidth;
        } else {
            ratio = transformedPageHeight / imageHeight;
        }
        int targetWidth = (int) (imageWidth * ratio);
        int targetHeight = (int) (imageHeight * ratio);

        int fromLeft = (pageWidth - targetWidth) / 2;
        int fromBottom = (pageHeight - targetHeight) / 2;

        mCurrentPage.addImage(fromLeft, fromBottom, targetWidth, targetHeight, rotationDegrees, img);
    }

    /**
     * Adds text annotation to current page of PDF document
     * @param text Annotation content
     * @param rect coordinates of clickable area to open popup with annotation
     */
    public void addAnnotation(String text, Rect rect) {
        mCurrentPage.addAnnotation(text, rect);
    }

    public String asString() {
        mPages.render();
        return mDocument.toPDFString();
    }

    public void write(OutputStream out) throws IOException {
        mPages.render();
        mDocument.write(out);
    }
}
