//
//  Android PDF Writer
//  http://coderesearchlabs.com/androidpdfwriter
//
//  by Javier Santo Domingo (j-a-s-d@coderesearchlabs.com)
//

package crl.android.pdfwriter;

import android.graphics.Matrix;
import android.graphics.Rect;

import java.util.ArrayList;

public class Page {

    private PDFDocument mDocument;
    private IndirectObject mIndirectObject;
    private ArrayList<IndirectObject> mPageFonts;
    private ArrayList<XObjectImage> mXObjects;
    private ArrayList<Annotation> mAnnotations;
    private IndirectObject mPageContents;
    private Array mMediaBox;

    public Page(PDFDocument document, int pageWidth, int pageHeight) {
        mMediaBox = new Array();
        if (pageWidth > 0 && pageWidth > 0) {
            String content[] = {"0", "0", Integer.toString(pageWidth), Integer.toString(pageHeight)};
            mMediaBox.addItemsFromStringArray(content);
        }

        mDocument = document;
        mIndirectObject = mDocument.newIndirectObject();
        mPageFonts = new ArrayList<IndirectObject>();
        mXObjects = new ArrayList<XObjectImage>();
        mAnnotations = new ArrayList<Annotation>();
        setFont(StandardFonts.SUBTYPE, StandardFonts.TIMES_ROMAN, StandardFonts.WIN_ANSI_ENCODING);
        mPageContents = mDocument.newIndirectObject();
        mDocument.includeIndirectObject(mPageContents);
    }

    public IndirectObject getIndirectObject() {
        return mIndirectObject;
    }

    private String getFontReferences() {
        String result = "";
        if (!mPageFonts.isEmpty()) {
            result = "    /Font <<\n";
            int x = 0;
            for (IndirectObject lFont : mPageFonts) {
                result += "      /F" + Integer.toString(++x) + " " + lFont.getIndirectReference() + "\n";
            }
            result += "    >>\n";
        }
        return result;
    }

    private String getAnnotationReferences() {
        String result = "";
        if (!mAnnotations.isEmpty()) {
            result = "  /Annots [ ";
            for (Annotation annotation : mAnnotations) {
                result += annotation.getIndirectObject().getIndirectReference() + " ";
            }
            result += "]\n";
        }
        return result;
    }

    private String getXObjectReferences() {
        String result = "";
        if (!mXObjects.isEmpty()) {
            result = "    /XObject <<\n";
            for (XObjectImage xObj : mXObjects) {
                result += "      " + xObj.asXObjectReference() + "\n";
            }
            result += "    >>\n";
        }
        return result;
    }

    public void render(String pagesIndirectReference) {
        String mediaBox = mMediaBox.hasContent() ? "  /MediaBox " + mMediaBox.toPDFString() + "\n" : "";

        mIndirectObject.setDictionaryContent(
                "  /Type /Page\n  /Parent " + pagesIndirectReference + "\n" +
                        mediaBox +
                        "  /Resources <<\n" + getFontReferences() + getXObjectReferences() + "  >>\n" +
                        "  /Contents " + mPageContents.getIndirectReference() + "\n" +
                        getAnnotationReferences()
        );
    }

    public void setFont(String subType, String baseFont) {
        IndirectObject lFont = mDocument.newIndirectObject();
        mDocument.includeIndirectObject(lFont);
        lFont.setDictionaryContent("  /Type /Font\n  /Subtype /" + subType + "\n  /BaseFont /" + baseFont + "\n");
        mPageFonts.add(lFont);
    }

    public void setFont(String subType, String baseFont, String encoding) {
        IndirectObject lFont = mDocument.newIndirectObject();
        mDocument.includeIndirectObject(lFont);
        lFont.setDictionaryContent("  /Type /Font\n  /Subtype /" + subType + "\n  /BaseFont /" + baseFont + "\n  /Encoding /" + encoding + "\n");
        mPageFonts.add(lFont);
    }

    private void addContent(String content) {
        mPageContents.addStreamContent(content);
    }

    public void addRawContent(String rawContent) {
        addContent(rawContent);
    }

    public void addText(int leftPosition, int topPositionFromBottom, int fontSize, String text) {
        addText(leftPosition, topPositionFromBottom, fontSize, text, Transformation.DEGREES_0_ROTATION);
    }

    public void addText(int leftPosition, int topPositionFromBottom, int fontSize, String text, String transformation) {
        addContent(
                "BT\n" +
                        transformation + " " + Integer.toString(leftPosition) + " " + Integer.toString(topPositionFromBottom) + " Tm\n" +
                        "/F" + Integer.toString(mPageFonts.size()) + " " + Integer.toString(fontSize) + " Tf\n" +
                        "(" + text + ") Tj\n" +
                        "ET\n"
        );
    }

    public void addLine(int fromLeft, int fromBottom, int toLeft, int toBottom) {
        addContent(
                Integer.toString(fromLeft) + " " + Integer.toString(fromBottom) + " m\n" +
                        Integer.toString(toLeft) + " " + Integer.toString(toBottom) + " l\nS\n"
        );
    }

    public void addRectangle(int fromLeft, int fromBottom, int toLeft, int toBottom) {
        addContent(
                Integer.toString(fromLeft) + " " + Integer.toString(fromBottom) + " " +
                        Integer.toString(toLeft) + " " + Integer.toString(toBottom) + " re\nS\n"
        );
    }

    private String ensureXObjectImage(XObjectImage xObject) {
        for (XObjectImage x : mXObjects) {
            if (x.getId().equals(xObject.getId())) {
                return x.getName();
            }
        }
        mXObjects.add(xObject);
        xObject.appendToDocument();
        return xObject.getName();
    }

    /**
     * Adds image stream to the page. Doesn't loads image in memory and, instead, copies image stream directly into document during
     * saving.
     *
     * @param translationX    X translation from the left
     * @param translationY    Y translation from the bottom
     * @param width           width of displayable area
     * @param height          height of displayable area
     * @param rotationDegrees rotation in degrees (rotation performed using center of the document as a pivot point)
     * @param xImage
     */
    public void addImage(int translationX, int translationY, int width, int height, int rotationDegrees, XObjectImage xImage) {
        Matrix translationMatrix = new Matrix();
        translationMatrix.setTranslate(translationX, translationY);

        Matrix rotationMatrix = new Matrix();
        rotationMatrix.setRotate(rotationDegrees, width / 2, height / 2);

        Matrix scaleMatrix = new Matrix();
        scaleMatrix.setScale(width, height);

        final String name = ensureXObjectImage(xImage);

        addContent(
                "q\n" +
                        toPdfMatrix(translationMatrix) + " cm\n" +
                        toPdfMatrix(rotationMatrix) + " cm\n" +
                        toPdfMatrix(scaleMatrix) + " cm\n" +
                        name + " Do\n" +
                        "Q\n"
        );
    }

    private String toPdfMatrix(Matrix matrix) {
        final float[] points = new float[9];
        matrix.getValues(points);

        return (int) points[0] + " "
                + (int) points[3] + " "
                + (int) points[1] + " "
                + (int) points[4] + " "
                + (int) points[2] + " "
                + (int) points[5];

    }

    /**
     * Adds text annotation to PDF document
     * @param text Annotation content
     * @param rect coordinates of clickable area to open popup with annotation
     */
    public void addAnnotation(String text, Rect rect) {
        Annotation annotation = new Annotation(text, rect, mDocument);
        mAnnotations.add(annotation);
    }
}
