package net.doo.snap.lib.detector;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Point;
import android.graphics.PointF;

import com.github.rjeschke.neetutils.dispose.Disposable;
import com.github.rjeschke.neetutils.dispose.Disposer;

import java.io.IOException;
import java.util.List;

/**
 * EdgeDetectionSuite JNI wrapper class.
 *
 * @author René Jeschke (rene_jeschke@yahoo.de)
 */
public class ContourDetector implements Disposable {
    private final Disposer disposer;
    private final long pointer;
    private final static int[] COLOR_CLAMP_RED = new int[256 + 512];
    private final static int[] COLOR_CLAMP_GREEN = new int[256 + 512];
    private final static int[] COLOR_CLAMP_BLUE = new int[256 + 512];

    public final static int IMAGE_FILTER_NONE = 0;
    public final static int IMAGE_FILTER_COLOR_ENHANCED = 1;
    public final static int IMAGE_FILTER_GRAY = 2;
    public final static int IMAGE_FILTER_BINARIZED = 3;

    static {
        try {
            System.loadLibrary("scanbot-detector");
            registerNatives();
        } catch (final UnsatisfiedLinkError e) {
            // ignore exception here for the Robolectric tests
        }

        for (int i = -256; i < 512; i++) {
            final int v = i < 0 ? 0 : i > 255 ? 255 : i;
            COLOR_CLAMP_RED[i + 256] = 0xff000000 | (v << 16);
            COLOR_CLAMP_GREEN[i + 256] = v << 8;
            COLOR_CLAMP_BLUE[i + 256] = v;
        }
    }

    public ContourDetector() {
        this.pointer = ctor();
        this.disposer = new Disposer(this, new ContourDetectorDisposer(this.pointer));
        if (this.pointer == 0) {
            throw new IllegalStateException("Failed to create native resources.");
        }
    }

    /**
     * Tries to detect a polygon from the given image.
     *
     * @param nv21   NV21 encoded camera image
     * @param width  Width of the image pixels
     * @param height Height of the image in pixels
     * @return A {@link DetectionResult} that gives information about the recognition state.
     */
    public final DetectionResult detect(final byte[] nv21, final int width, final int height) {
        return detect(this.pointer, nv21, width, height);
    }

    /**
     * Tries to detect a polygon from the given image.
     *
     * @param data Encoded image (e.g. a JPEG)
     * @return A {@link DetectionResult} that gives information about the recognition state.
     */
    public final DetectionResult detect(final byte[] data) {
        final Bitmap bitmap = createBitmapFromEncodedData(data);
        final DetectionResult ret = detect(this.pointer, bitmap);
        bitmap.recycle();
        return ret;
    }

    /**
     * Tries to detect a polygon from the given file.
     *
     * @param fileName Encoded file path (e.g. a JPEG)
     * @return A {@link DetectionResult} that gives information about the recognition state.
     */
    public final DetectionResult detect(final String fileName) throws IOException {
        final Bitmap bitmap = createBitmapFromEncodedFile(fileName);
        final DetectionResult ret = detect(this.pointer, bitmap);
        bitmap.recycle();
        return ret;
    }

    /**
     * Tries to detect a polygon from the given image.
     *
     * @param bitmap The image
     * @return A {@link DetectionResult} that gives information about the recognition state.
     */
    public final DetectionResult detect(final Bitmap bitmap) {
        return detect(this.pointer, bitmap);
    }

    /**
     * Post processes an image, applying cropping and filters.
     *
     * @param nv21        NV21 encoded camera image
     * @param width       Width of the image pixels
     * @param height      Height of the image in pixels
     * @param polygon     The detected polygon
     * @param imageFilter Image filter type, one of {@code IMAGE_FILTER_*} or {@code 0}.
     * @return The processed Bitmap.
     */
    public final Bitmap processImage(final byte[] nv21, final int width, final int height, final List<Point> polygon, final int imageFilter) {
        final Bitmap bitmap = createBitmapFromNV21(nv21, width, height);
        final Bitmap ret = processImage(this.pointer, bitmap, polygon, imageFilter);
        bitmap.recycle();
        return ret;
    }

    /**
     * Post processes an image, applying cropping and filters.
     *
     * @param data        Encoded image (e.g. a JPEG)
     * @param polygon     The detected polygon
     * @param imageFilter Image filter type, one of {@code IMAGE_FILTER_*} or {@code 0}.
     * @return The processed Bitmap.
     */
    public final Bitmap processImage(final byte[] data, final List<Point> polygon, final int imageFilter) {
        final Bitmap bitmap = createBitmapFromEncodedData(data);
        final Bitmap ret = processImage(this.pointer, bitmap, polygon, imageFilter);
        bitmap.recycle();
        return ret;
    }

    /**
     * Post processes an image, applying cropping and filters.
     *
     * @param bitmap      The image to process
     * @param polygon     The detected polygon
     * @param imageFilter Image filter type, one of {@code IMAGE_FILTER_*} or {@code 0}.
     * @return The processed Bitmap.
     */
    public final Bitmap processImage(final Bitmap bitmap, final List<Point> polygon, final int imageFilter) {
        return processImage(this.pointer, bitmap, polygon, imageFilter);
    }

    /**
     * Post processes an image, applying cropping and filters.
     *
     * @param fileName    path to JPEG file
     * @param polygon     The detected polygon
     * @param imageFilter Image filter type, one of {@code IMAGE_FILTER_*} or {@code 0}.
     * @return The processed Bitmap.
     */
    public Bitmap processImageF(final String fileName, final List<PointF> polygon, final int imageFilter) throws IOException {
        final Bitmap bitmap = createBitmapFromEncodedFile(fileName);
        final Bitmap ret = processImageF(this.pointer, bitmap, polygon, imageFilter);
        bitmap.recycle();
        return ret;
    }

    /**
     * Post processes an image (using floating point coordinates), applying cropping and filters.
     *
     * @param nv21        NV21 encoded camera image
     * @param width       Width of the image pixels
     * @param height      Height of the image in pixels
     * @param polygon     The detected polygon
     * @param imageFilter Image filter type, one of {@code IMAGE_FILTER_*} or {@code 0}.
     * @return The processed Bitmap.
     */
    public final Bitmap processImageF(final byte[] nv21, final int width, final int height, final List<PointF> polygon, final int imageFilter) {
        final Bitmap bitmap = createBitmapFromNV21(nv21, width, height);
        final Bitmap ret = processImageF(this.pointer, bitmap, polygon, imageFilter);
        bitmap.recycle();
        return ret;
    }

    /**
     * Post processes an image (using floating point coordinates), applying cropping and filters.
     *
     * @param data        Encoded image (e.g. a JPEG)
     * @param polygon     The detected polygon
     * @param imageFilter Image filter type, one of {@code IMAGE_FILTER_*} or {@code 0}.
     * @return The processed Bitmap.
     */
    public final Bitmap processImageF(final byte[] data, final List<PointF> polygon, final int imageFilter) {
        final Bitmap bitmap = createBitmapFromEncodedData(data);
        final Bitmap ret = processImageF(this.pointer, bitmap, polygon, imageFilter);
        bitmap.recycle();
        return ret;
    }

    /**
     * Post processes an image (using floating point coordinates), applying cropping and filters.
     *
     * @param bitmap      The image to process
     * @param polygon     The detected polygon
     * @param imageFilter Image filter type, one of {@code IMAGE_FILTER_*} or {@code 0}.
     * @return The processed Bitmap.
     */
    public final Bitmap processImageF(final Bitmap bitmap, final List<PointF> polygon, final int imageFilter) {
        return processImageF(this.pointer, bitmap, polygon, imageFilter);
    }

    /**
     * @return the detected polygon (if any) as absolute pixel coordinates.
     */
    public final List<Point> getPolygon() {
        return getPolygon(this.pointer);
    }

    /**
     * @return the detected polygon (if any) as floating point coordinates in {@code [0.f, 1.f[}.
     */
    public final List<PointF> getPolygonF() {
        return getPolygonF(this.pointer);
    }

    /**
     * @return Detected vertical lines
     */
    public final List<Line2D> getVerticalLines() {
        return getVerticalLines(this.pointer);
    }

    /**
     * @return Detected horizontal lines
     */
    public final List<Line2D> getHorizontalLines() {
        return getHorizontalLines(this.pointer);
    }

    /**
     * Calling this method will reset the polygon smoothing history.
     */
    public final void resetPolygonHistory() {
        resetPolygonHistory(this.pointer);
    }

    /**
     * Disposes the native resources held by this instance.
     */
    @Override
    public void dispose() {
        this.disposer.dispose();
    }

    /**
     * Creates a {@code ARGB_8888} Bitmap from an in-memory JPEG.
     *
     * @param data JPEG data.
     * @return The decoded Bitmap.
     */
    public final static Bitmap createBitmapFromEncodedData(final byte[] data) {
        final BitmapFactory.Options options = new BitmapFactory.Options();
        options.inDither = false;
        options.inPreferredConfig = Bitmap.Config.ARGB_8888;
        return BitmapFactory.decodeByteArray(data, 0, data.length, options);
    }

    /**
     * Creates a {@code ARGB_8888} Bitmap from encoded JPEG on file system
     *
     * @param fileName path to JPEG file
     * @return The decoded Bitmap.
     */
    public final static Bitmap createBitmapFromEncodedFile(final String fileName) throws IOException {
        final BitmapFactory.Options options = new BitmapFactory.Options();
        options.inDither = false;
        options.inPreferredConfig = Bitmap.Config.ARGB_8888;
        Bitmap bitmap = BitmapFactory.decodeFile(fileName, options);
        if (bitmap == null) {
            throw new IOException("Can't open Bitmap");
        }

        return bitmap;
    }

    /**
     * Converts a NV21 encoded image into a {@code ARGB_8888} Bitmap (using fixed point arithmetics).
     *
     * @param nv21   Bytes to decode.
     * @param width  The width of the image.
     * @param height The height of the image.
     * @return The decoded image as a Bitmap.
     */
    public final static Bitmap createBitmapFromNV21(final byte[] nv21, final int width, final int height) {
        final int yStride = (width + 15) & ~15;
        final int[] image = new int[width * height];
        final int h2 = height >> 1;

        for (int y = 0; y < h2; y++) {
            final int yOffs = (y << 1) * yStride;
            final int vuOffs = (height + y) * yStride;
            final int oOffs = (y << 1) * width;

            for (int x = 0; x < width; x += 2) {
                final int v2 = (nv21[vuOffs + x] & 255) - 128;
                final int u2 = (nv21[vuOffs + x + 1] & 255) - 128;

                final int r0 = 91880 * v2;
                final int g0 = 22553 * u2 + 46802 * v2;
                final int b0 = 116130 * u2;

                final int yOffsX = yOffs + x;

                final int l00 = (256 + (nv21[yOffsX] & 255)) << 16;
                final int l10 = (256 + (nv21[yOffsX + 1] & 255)) << 16;
                final int l01 = (256 + (nv21[yOffsX + yStride] & 255)) << 16;
                final int l11 = (256 + (nv21[yOffsX + yStride + 1] & 255)) << 16;

                final int oOffsX = oOffs + x;

                image[oOffsX] = COLOR_CLAMP_RED[(l00 + r0) >> 16] | COLOR_CLAMP_GREEN[(l00 - g0) >> 16] | COLOR_CLAMP_BLUE[(l00 + b0) >> 16];
                image[oOffsX + 1] = COLOR_CLAMP_RED[(l10 + r0) >> 16] | COLOR_CLAMP_GREEN[(l10 - g0) >> 16] | COLOR_CLAMP_BLUE[(l10 + b0) >> 16];
                image[oOffsX + width] = COLOR_CLAMP_RED[(l01 + r0) >> 16] | COLOR_CLAMP_GREEN[(l01 - g0) >> 16] | COLOR_CLAMP_BLUE[(l01 + b0) >> 16];
                image[oOffsX + width + 1] = COLOR_CLAMP_RED[(l11 + r0) >> 16] | COLOR_CLAMP_GREEN[(l11 - g0) >> 16] | COLOR_CLAMP_BLUE[(l11 + b0) >> 16];
            }
        }

        return Bitmap.createBitmap(image, width, height, Bitmap.Config.ARGB_8888);
    }

    /**
     * Disposer wrapper.
     *
     * @author René Jeschke (rene_jeschke@yahoo.de)
     */
    private final static class ContourDetectorDisposer implements Disposable {
        private final long ptr;

        public ContourDetectorDisposer(final long ptr) {
            this.ptr = ptr;
        }

        @Override
        public void dispose() {
            ContourDetector.dtor(this.ptr);
        }
    }

    private static native long ctor();

    static native void dtor(long ptr);

    private static native void resetPolygonHistory(long ptr);

    private static native List<Point> getPolygon(long ptr);

    private static native List<PointF> getPolygonF(long ptr);

    private static native DetectionResult detect(long ptr, final byte[] nv21, int width, int height);

    private static native DetectionResult detect(long ptr, final Bitmap bitmap);

    private static native Bitmap processImage(long ptr, final Bitmap bitmap, final List<Point> polygon, final int processingMode);

    private static native Bitmap processImageF(long ptr, final Bitmap bitmap, final List<PointF> polygon, final int processingMode);

    private static native List<Line2D> getVerticalLines(long ptr);

    private static native List<Line2D> getHorizontalLines(long ptr);

    private static native void registerNatives();
}
