package io.scanbot.dynawrapper;

import android.graphics.Bitmap;

//TODO's
// CreateNewPDF
//  
// 
// SetDocInfo
// Append
// EndPage
// CloseFile
// SetSaveImageFormat
// InsertImageEx
// SetBBox

public class DynaPDF
{
    // Java doesn't support enums like almost every other programming language in this world. An enum in Java is an array of classes which
	// is totally idiotic defined. So, we can only use integer constansts if we don't want to write a separate program that is able to
	// identify a predefined constant!

    public static final int psFitWidth  = 0; // Scale the page to the width of the image buffer
    public static final int psFitHeight = 1; // Scale the page to the height of the image buffer
    public static final int psFitBest   = 2; // Scale the page so that it fits fully into the image buffer
    public static final int psFitZoom   = 3; // This mode should be used if the scaling factors of the transformation matrix are <> 1.0

    public static final int ptOpen        = 0;
    public static final int ptOwner       = 1;
    public static final int ptForceRepair = 2;

    public static final int rfDefault         = 0x00000000; // Render the page as usual
    public static final int rfScaleToMediaBox = 0x00000001; // Render the real paper format. Contents outside the crop box is clipped
    public static final int rfIgnoreCropBox   = 0x00000002; // Ignore the crop box and render anything inside the media box without clipping
   /*
      The art, bleed, or trim box is first intersected with the media box or crop box if present
      since these boxes represent the maximum extend of the page.
      If the flag rfIgnoreCropBox is set, the boxes are intersected with the media box. The page is
      scaled to the media or crop depending on the above flags.
   */
    public static final int rfClipToArtBox    = 0x00000004; // Clip the page to the art box if any
    public static final int rfClipToBleedBox  = 0x00000008; // Clip the page to the bleed box if any
    public static final int rfClipToTrimBox   = 0x00000010; // Clip the page to the trim box if any
    // ----------------------------
    public static final int rfExclAnnotations = 0x00000020; // Don't render annotations
    public static final int rfExclFormFields  = 0x00000040; // Don't render form fields
    public static final int rfSkipUpdateBG    = 0x00000080; // Don't generate an update event after initializing the background to white
    public static final int rfRotate90        = 0x00000100; // Rotate the page 90 degress
    public static final int rfRotate180       = 0x00000200; // Rotate the page 180 degress
    public static final int rfRotate270       = 0x00000400; // Rotate the page 270 degress

    private Bitmap m_Bitmap    = null;
    private int    m_DefScale  = psFitWidth;
    private int    m_Flags     = rfDefault;
    private long   m_Instance  = 0;
    private float  m_Scale     = 1.0f;

	public DynaPDF() throws PDFException
	{
		m_Instance = CreateInstance();
		if (m_Instance == 0) throw new OutOfMemoryError("Out of memory!");
	}

	public void onDestroy()
	{
		if (m_Instance != 0)
		{
			DeleteInstance(m_Instance);
			m_Instance = 0;
		}
	}

	public void abort()
	{
		Abort(m_Instance);
	}

	public Bitmap getBitmap()
	{
		return m_Bitmap;
	}

    public int openFile(String fileName, int pwdType, String password)
    {
    	m_Scale = 1.0f;
    	return OpenPDFFile(m_Instance, fileName, pwdType, password);
    }

    /**
     * Renders page with given 1-based position into {@link Bitmap} which later can be obtained with
     * {@link #getBitmap()}.
     * 
     * @param pageNum 1-based (so, first page is 1) page number to render
     * @param screenWidth frame width. {@code -1} to use page width.
     * @param screenHeight frame height. {@code -1} to use page height.
     * @return {@code true} on success, otherwise {@code false}
     */
	public boolean renderPage(int pageNum, int screenWidth, int screenHeight)
	{
		int[] size = new int[2];
		if (!PrepareForRender(m_Instance, pageNum, m_DefScale, m_Scale, screenWidth, screenHeight, m_Flags, size)) return false;
		if (m_Bitmap == null || m_Bitmap.getWidth() != size[0] || m_Bitmap.getHeight() != size[1])
			m_Bitmap = Bitmap.createBitmap(size[0], size[1], Bitmap.Config.ARGB_8888);
		return RenderPage(m_Instance, pageNum, 0, 0, m_Scale, m_DefScale, m_Bitmap);
	}
	
	public String[] getPageAnnotations(int pageNum){
		return null;
	}
	
	/**
	 * Creates a new PDF file on a given path and prepares it for writing.
	 * As soon as you'll finish with file, call {@link #closeFile()}.
	 * 
	 * @param filePath path to created PDF.
	 * @throws PDFException in case of native error
	 */
	public void createNewPdf(String filePath) throws PDFException {
		if (!CreateNewPdf(m_Instance, filePath)) {
			throw new PDFException();
		}
	}
	
	/**
	 * Set given value to document information field.
	 * 
	 * @param field document information field
	 * @param value value to set
	 * @throws PDFException in case of native error
	 */
	public void setDocumentInfoField(DocumentInfoField field, String value) throws PDFException {
		if (!SetDocInfo(m_Instance, field.ordinal(), value)) {
			throw new PDFException();
		}
	}
	
	/**
	 * Appends a new empty page to the document.
	 * 
	 * @throws PDFException in case of native error
	 */
	public void appendPage() throws PDFException {
		if (!Append(m_Instance)) {
			throw new PDFException();
		}
	}
	
	/**
	 * Assigns a bounding box to current page.
	 * 
	 * @param pageBoundary type of page boundary. See DynaPDF documentation for more info (SetBBox).
	 * @param left
	 * @param top
	 * @param right
	 * @param bottom
	 * @throws PDFException in case of native error
	 */
	public void setBoundingBox(PageBoundary pageBoundary, double left, double top, double right, double bottom) throws PDFException {
		if (!SetBBox(m_Instance, pageBoundary.ordinal(), left, top, right, bottom)) {
			throw new PDFException();
		}
	}
	
	/**
	 * Inserts image from a given file path into current page.
	 * See DynaPDF and PDF specifications for explanation of arguments.
	 * 
	 * @throws PDFException in case of native error
	 */
	public void insertImage(double x, double y, double scaleWidth, double scaleHeight, String filePath) throws PDFException {
		if (!InsertImageEx(m_Instance, x, y, scaleWidth, scaleHeight, filePath)) {
			throw new PDFException();
		}
	}
	
	/**
	 * Left multiplies current transformation matrix of current page with given matrix.
	 * Consider calling {@link #saveGraphicsState()} before transformations and {@link #restoreGraphicsState()}
	 * when you finished with current transformation matrix, so it will become identity matrix once again.
	 * 
	 * @throws PDFException in case of native error
	 */
	public void applyMatrix(double a, double b, double c, double d, double x, double y) throws PDFException {
		if (!SetMatrix(m_Instance, a, b, c, d, x, y)) {
			throw new PDFException();
		}
	}
	
	/**
	 * Saves current graphical state (including current transformation matrix).
	 * States might be nested.
	 * 
	 * @throws PDFException in case of native error
	 */
	public void saveGraphicsState() throws PDFException {
		if (!SaveGraphicsState(m_Instance)) {
			throw new PDFException();
		}
	}
	
	/**
	 * Restores graphical state to the previous invocation of {@link #saveGraphicsState()}
	 *  
	 * @throws PDFException in case of native error
	 */
	public void restoreGraphicsState() throws PDFException {
		if (!RestoreGraphicsState(m_Instance)) {
			throw new PDFException();
		}
	}
	
	/**
	 * Finish writing of current page
	 * 
	 * @throws PDFException in case of native error
	 */
	public void endPage() throws PDFException {
		if (!EndPage(m_Instance)) {
			throw new PDFException();
		}
	}
	
	/**
	 * Finish writing of the document. PDF document will be written to the file system.
	 * 
	 * @throws PDFException in case of native error
	 */
	public void closeFile() throws PDFException {
		if (!CloseFile(m_Instance)) {
			throw new PDFException();
		}
	}
	
	/**
	 * Imports currently opened PDF file, so it can be edited.
	 * 
	 * @param destPage specifies the destination page on which the new pages will be inserted or added if the pages already exist
	 * @param scaleX horizontal scaling factor (1.0 = no scaling)
	 * @param scaleY vertical scaling factor (1.0 = no scaling)
	 * 
	 * @throws PDFException in case of native error
	 */
	public void importPDF(int destPage, double scaleX, double scaleY) throws PDFException {
		if (!ImportPDFFile(m_Instance, destPage, scaleX, scaleY)) {
			throw new PDFException();
		}
	}
	
	/**
	 * Closes previously opened for import file
	 */
	public void closeImportedFile() {
		CloseImportFile(m_Instance);
	}
	
	/**
	 * Prepares page with given number for editing.
	 * 
	 * @param pageNumber 1-based page number
	 * 
	 * @throws PDFException in case of native error
	 */
	public void editPage(int pageNumber) throws PDFException {
		if (!EditPage(m_Instance, pageNumber)) {
			throw new PDFException();
		}
	}
	
	/**
	 * Creates a text PDF annotation
	 * 
	 * @param posX X-Coordinate of the annotation
	 * @param posY Y-Coordinate of the annotation
	 * @param width width of the window in open state
	 * @param height height of the window in open state
	 * @param author author of the annotation or {@code null}
	 * @param text the annotation's Text or {@code null}
	 * @param icon annotation icon
	 * @param open Should the annotation appear open or closed?
	 * @return annotation handle
	 * 
	 * @throws PDFException in case of native error
	 */
	public int createTextAnnotation(
			double posX,
			double posY,
			double width,
			double height,
			String author,
			String text,
			AnnotationIcon icon,
			boolean open
	) throws PDFException{
		
		int annotationHandle = TextAnnot(m_Instance, posX, posY, width, height, author, text, icon.ordinal(), open);
		if (annotationHandle < 0) {
			throw new PDFException();
		}
		
		return annotationHandle;
	}
	
	/**
	 * Assigns output file to in-memory PDF (which is the case when PDF opened for editing).
	 * 
	 * @param path output path
	 * 
	 * @throws PDFException in case of native error
	 */
	public void openOutputFile(String path) throws PDFException {
		if (!OpenOutputFile(m_Instance, path)) {
			throw new PDFException();
		}
	}
	
	/**
	 * @return width of current page
	 */
	public int getCurrentPageWidth() {
		return GetPageWidth(m_Instance);
	}
	
	/**
	 * @return height of current page
	 */
	public int getCurrentPageHeight() {
		return GetPageHeight(m_Instance);
	}

	private static native void    Abort(long InsPtr);
	private static native long    CreateInstance();
	private static native void    DeleteInstance(long InsPtr);
	private static native boolean PrepareForRender(long InstPtr, int PageNum, int DefScale, float Scale, int FrameWidth, int FrameHeight, int Flags, int[] Size);
	private static native int     OpenPDFFile(long InstPtr, String FileName, int PwdType /* TPwdType */, String Password);
	private static native boolean RenderPage(long InstPtr, int PageNum, int X, int Y, float Scale, int DefScale, Bitmap bitmap);
	
	/*
	 * Methods for PDF creation
	 */
	private static native boolean CreateNewPdf(long instancePointer, String filePath);
	private static native boolean SetDocInfo(long instancePointer, int tDocumentInfo, String value);
	private static native boolean Append(long instancePointer);
	private static native boolean SetBBox(long instancePointer, int tPageBoundary, double leftX, double leftY, double rightX, double rightY);
	private static native boolean InsertImageEx(long instancePointer, double posX, double posY, double scaleWidth, double scaleHeight, String filePath);
	private static native boolean SetMatrix(long instancePointer, double a, double b, double c, double d, double x, double y);
	private static native boolean SaveGraphicsState(long instancePointer);
	private static native boolean RestoreGraphicsState(long instancePointer);
	private static native boolean EndPage(long instancePointer);
	private static native boolean CloseFile(long instancePointer);
	
	/*
	 * Methods for PDF editing
	 */
	private static native boolean ImportPDFFile(long instancePointer, int destPage, double scaleX, double scaleY);
	private static native int TextAnnot(long instancePointer, double posX, double posY, double width, double height, String author, String text, int tAnnotIcon, boolean open);
	private static native void CloseImportFile(long instancePointer);
	private static native boolean EditPage(long instancePointer, int pageNumber);
	private static native boolean OpenOutputFile(long instancePointer, String path);
	private static native int GetPageWidth(long instancePointer);
	private static native int GetPageHeight(long instancePointer);
	
	static
	{
		try {
			System.loadLibrary("dynapdf");
		} catch(UnsatisfiedLinkError e) {
			/*
			 * Do nothing. We need it to properly execute tests on Robolectric
			 */
		}
	}
}
