package net.doo.snap.lib.ui.util;

import android.view.MotionEvent;
import android.view.View;

/**
 * Animates swipe-down gesture on touch events. By the end of touch event makes a decision whether
 * swipe-down was performed or not and either restores view transformation or invokes {@link #onSwipeDownDetected()}
 * <p/>
 * Holds reference to {@link android.view.View}.
 */
public abstract class SwipeDownTouchListener implements View.OnTouchListener {

    private static final float SWIPE_DOWN_FACTOR = 0.3f;
    private final View view;
    private float historicalY;
    private float diff;

    protected SwipeDownTouchListener(View view) {
        this.view = view;
    }

    @Override
    public boolean onTouch(View v, MotionEvent event) {
        final float y = event.getY() + view.getTranslationY();

        switch (event.getAction()) {
            case MotionEvent.ACTION_DOWN:
                historicalY = y;
                diff = 0f;
                break;
            case MotionEvent.ACTION_MOVE:
                diff = y - historicalY;

                if (diff < 0f) {
                    diff = 0f;
                }

                if (diff > view.getHeight()) {
                    onSwipeDownDetected();
                    return false;
                }

                view.setTranslationY(diff);
                view.setAlpha((view.getHeight() - diff) / (float) view.getHeight());

                break;
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                if (diff > view.getHeight() * SWIPE_DOWN_FACTOR) {
                    onSwipeDownDetected();
                } else {
                    view.animate()
                            .alpha(ViewUtils.ALPHA_DEFAULT)
                            .translationY(ViewUtils.TRANSLATION_DEFAULT)
                            .setListener(new ViewUtils.HardwareLayerAnimationListener(view));
                }

                break;
        }

        return true;
    }

    /**
     * Invoked when swipe-down gesture was detected
     */
    protected abstract void onSwipeDownDetected();

}
