package net.doo.snap.lib.ui.util;

import android.annotation.TargetApi;
import android.content.res.Resources;
import android.graphics.Canvas;
import android.graphics.ColorFilter;
import android.graphics.Matrix;
import android.graphics.PorterDuff;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.Region;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.util.AttributeSet;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import java.io.IOException;

/**
 * Wraps another {@link android.graphics.drawable.Drawable} and provide a way to perform transformations
 * on it.
 */
public class TransformableDrawable extends Drawable {

    public static final String PROPERTY_ROTATION = "rotation";
    public static final String PROPERTY_SCALE = "scale";

    private Drawable wrappedDrawable;

    private float rotation = 0f;
    private float scale = ViewUtils.SCALE_DEFAULT;

    private boolean adjustBounds = false;
    private final Matrix matrix = new Matrix();
    private final RectF transformedBounds = new RectF();

    public TransformableDrawable(Drawable wrappedDrawable) throws NullPointerException {
        if (wrappedDrawable == null) {
            throw new NullPointerException("Wrapped drawable can't be null");
        }

        this.wrappedDrawable = wrappedDrawable;
    }

    /**
     * @return current rotation of the drawable
     */
    public float getRotation() {
        return rotation;
    }

    /**
     * @param rotation of the drawable
     */
    public void setRotation(float rotation) {
        this.rotation = rotation;

        if (adjustBounds) {
            transformedBounds.set(0f, 0f, 0f, 0f);
        }

        invalidateSelf();
    }

    /**
     * @return current scale of the drawable
     */
    public float getScale() {
        return scale;
    }

    /**
     * @param scale of the drawable
     */
    public void setScale(float scale) {
        this.scale = scale;

        if (adjustBounds) {
            transformedBounds.set(0f, 0f, 0f, 0f);
        }

        invalidateSelf();
    }

    /**
     * If {@code true} {@link net.doo.snap.lib.ui.util.TransformableDrawable} will recalculate it's width
     * and height according to given transformation. {@code false} will keep width and height of wrapped {@link android.graphics.drawable.Drawable}.
     * Default is {@code false}.
     */
    public void setAdjustBounds(boolean adjustBounds) {
        this.adjustBounds = adjustBounds;
        transformedBounds.set(0f, 0f, 0f, 0f);
        invalidateSelf();
    }

    @Override
    public void draw(Canvas canvas) {
        Rect bounds = getBounds();
        final int halfWidth = bounds.centerX();
        final int halfHeight = bounds.centerY();

        canvas.save();
        canvas.scale(
                scale, scale,
                halfWidth, halfHeight

        );
        canvas.rotate(rotation, halfWidth, halfHeight);

        wrappedDrawable.draw(canvas);

        canvas.restore();
    }

    @Override
    public void setBounds(int left, int top, int right, int bottom) {
        super.setBounds(left, top, right, bottom);

        if (!adjustBounds) {
            wrappedDrawable.setBounds(left, top, right, bottom);
        } else {
            int x = left + (right - left) / 2 - wrappedDrawable.getIntrinsicWidth() / 2;
            int y = top + (bottom - top) / 2 - wrappedDrawable.getIntrinsicHeight() / 2;
            wrappedDrawable.setBounds(
                    x, y,
                    x + wrappedDrawable.getIntrinsicWidth(),
                    y + wrappedDrawable.getIntrinsicHeight()
            );
        }
    }

    @Override
    public void setBounds(Rect bounds) {
        super.setBounds(bounds);

        if (!adjustBounds) {
            wrappedDrawable.setBounds(bounds);
        } else {
            int x = bounds.left + bounds.centerX() - wrappedDrawable.getIntrinsicWidth() / 2;
            int y = bounds.top + bounds.centerY() - wrappedDrawable.getIntrinsicHeight() / 2;
            wrappedDrawable.setBounds(
                    x, y,
                    x + wrappedDrawable.getIntrinsicWidth(),
                    y + wrappedDrawable.getIntrinsicHeight()
            );
        }
    }

    @Override
    public void setChangingConfigurations(int configs) {
        wrappedDrawable.setChangingConfigurations(configs);
    }

    @Override
    public int getChangingConfigurations() {
        return wrappedDrawable.getChangingConfigurations();
    }

    @Override
    public void setDither(boolean dither) {
        wrappedDrawable.setDither(dither);
    }

    @Override
    public void setFilterBitmap(boolean filter) {
        wrappedDrawable.setFilterBitmap(filter);
    }

    @Override
    public void setAlpha(int alpha) {
        wrappedDrawable.setAlpha(alpha);
    }

    @TargetApi(Build.VERSION_CODES.KITKAT)
    @Override
    public int getAlpha() {
        return wrappedDrawable.getAlpha();
    }

    @Override
    public void setColorFilter(ColorFilter cf) {
        wrappedDrawable.setColorFilter(cf);
    }

    @Override
    public void setColorFilter(int color, PorterDuff.Mode mode) {
        wrappedDrawable.setColorFilter(color, mode);
    }

    @Override
    public void clearColorFilter() {
        wrappedDrawable.clearColorFilter();
    }

    @Override
    public boolean isStateful() {
        return wrappedDrawable.isStateful();
    }

    @Override
    public boolean setState(int[] stateSet) {
        return wrappedDrawable.setState(stateSet);
    }

    @Override
    public int[] getState() {
        return wrappedDrawable.getState();
    }

    @TargetApi(Build.VERSION_CODES.HONEYCOMB)
    @Override
    public void jumpToCurrentState() {
        wrappedDrawable.jumpToCurrentState();
    }

    @Override
    public Drawable getCurrent() {
        return wrappedDrawable.getCurrent();
    }

    @Override
    public boolean setVisible(boolean visible, boolean restart) {
        return wrappedDrawable.setVisible(visible, restart);
    }

    @Override
    public int getOpacity() {
        return wrappedDrawable.getOpacity();
    }

    @Override
    public Region getTransparentRegion() {
        return wrappedDrawable.getTransparentRegion();
    }

    @Override
    public int getIntrinsicWidth() {
        if (!adjustBounds) {
            return wrappedDrawable.getIntrinsicWidth();
        }

        if (transformedBounds.isEmpty()) {
            recalculateBounds();
        }

        return (int) transformedBounds.width();
    }

    @Override
    public int getIntrinsicHeight() {
        if (!adjustBounds) {
            return wrappedDrawable.getIntrinsicHeight();
        }

        if (transformedBounds.isEmpty()) {
            recalculateBounds();
        }

        return (int) transformedBounds.height();
    }

    private void recalculateBounds() {
        final float px = wrappedDrawable.getIntrinsicWidth() / 2f;
        final float py = wrappedDrawable.getIntrinsicHeight() / 2f;

        matrix.setScale(scale, scale, px, py);
        matrix.preRotate(rotation, px, py);

        transformedBounds.set(0f, 0f, wrappedDrawable.getIntrinsicWidth(), wrappedDrawable.getIntrinsicHeight());
        matrix.mapRect(transformedBounds);
    }

    @Override
    public int getMinimumWidth() {
        return 0;
    }

    @Override
    public int getMinimumHeight() {
        return 0;
    }

    @Override
    public boolean getPadding(Rect padding) {
        return wrappedDrawable.getPadding(padding);
    }

    @Override
    public Drawable mutate() {
        return this;
    }

    @Override
    public void inflate(Resources r, XmlPullParser parser, AttributeSet attrs) throws XmlPullParserException, IOException {
        wrappedDrawable.inflate(r, parser, attrs);
    }

    @Override
    public ConstantState getConstantState() {
        return wrappedDrawable.getConstantState();
    }
}
