package net.doo.snap.lib.ui;

import android.app.ActionBar;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.os.AsyncTask;
import android.os.Bundle;
import android.support.v4.app.LoaderManager;
import android.support.v4.content.Loader;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.widget.SeekBar;

import com.google.inject.Inject;

import net.doo.snap.lib.R;
import net.doo.snap.lib.analytics.ActivityAnalytics;
import net.doo.snap.lib.persistence.Page;
import net.doo.snap.lib.persistence.PageStoreStrategy;
import net.doo.snap.lib.persistence.Signature;
import net.doo.snap.lib.persistence.SignatureStoreStrategy;
import net.doo.snap.lib.sensor.SensorHelper;
import net.doo.snap.lib.sensor.ShakeListener;
import net.doo.snap.lib.sensor.SignificantMoveListener;
import net.doo.snap.lib.snap.util.OrientationLocker;
import net.doo.snap.lib.ui.util.ViewUtils;
import net.doo.snap.lib.ui.widget.AlignBlockImageView;
import net.doo.snap.lib.util.bitmap.BitmapLruCache;
import net.doo.snap.lib.util.bitmap.BitmapUtils;
import net.doo.snap.lib.util.bitmap.PageBitmapLoader;
import net.doo.snap.lib.util.log.DebugLog;

import java.io.IOException;

import roboguice.activity.RoboFragmentActivity;

/**
 * Provides workflow for editing (moving/creating/removing) of page's signature
 */
public class EditSignatureActivity extends RoboFragmentActivity implements LoaderManager.LoaderCallbacks<Bitmap> {

    public static final String ARGUMENT_PAGE = "page";
    public static final String RESULT_SIGNATURE = "signature";

    private static final int CREATE_SIGNATURE_REQUEST_CODE = 100;
    private static final float MIN_SCALE = 0.3f;
    private static final int DEFAULT_SCALE_PROGRESS = 50;
    private static final String STATE_SIGNATURE = "state_signature";
    private static final String LOADER_PATH = "path";

    @Inject
    private OrientationLocker orientationLocker;
    @Inject
    private PageStoreStrategy pageStoreStrategy;
    @Inject
    private SignatureStoreStrategy signatureStoreStrategy;
    @Inject
    private ActivityAnalytics activityAnalytics;
    @Inject
    private BitmapLruCache bitmapLruCache;
    @Inject
    private SensorHelper sensorHelper;

    private Page page;
    private Signature signature;
    private Bitmap signatureBitmap;

    private AlignBlockImageView imageView;

    private MenuItem addSignature;
    private MenuItem editSignature;
    private MenuItem clearSignature;
    private SeekBar sizeBar;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_edit_signature);

        page = getIntent().getParcelableExtra(ARGUMENT_PAGE);
        if (page == null) {
            throw new NullPointerException("Page can't be null");
        }

        ActionBar actionBar = getActionBar();
        actionBar.setDisplayShowHomeEnabled(false);
        actionBar.setDisplayShowTitleEnabled(false);
        actionBar.setDisplayShowCustomEnabled(true);
        actionBar.setDisplayHomeAsUpEnabled(false);
        actionBar.setCustomView(R.layout.done);

        imageView = (AlignBlockImageView) findViewById(R.id.image);

        sizeBar = (SeekBar) findViewById(R.id.size_seek_bar);
        sizeBar.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {

            private int lastProgress = DEFAULT_SCALE_PROGRESS;

            @Override
            public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser) {
                if (!fromUser) {
                    return;
                }

                final float scale = progressToScale(progress);

                if (imageView.setBlockScale(scale)) {
                    lastProgress = progress;
                } else {
                    seekBar.setProgress(lastProgress);
                }
            }

            @Override
            public void onStartTrackingTouch(SeekBar seekBar) {
            }

            @Override
            public void onStopTrackingTouch(SeekBar seekBar) {
            }

        });

        findViewById(R.id.done).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                deliverResult();
            }
        });

        if (savedInstanceState == null) {
            signature = page.getSignature();

            if (signature == null) {
                requestSignature();
            }
        } else {
            restoreState(savedInstanceState);
        }

        try {
            getSupportLoaderManager().initLoader(0, buildLoaderArgs(), this);
        } catch (IOException e) {
            DebugLog.logException(e);
        }

        sensorHelper.registerSignificantMoveListener(new ShakeListener() {
            @Override
            public void onShake() {
                resetSignature();
            }
        });
    }

    private void restoreState(Bundle savedInstanceState) {
        if (savedInstanceState == null) {
            return;
        }

        signature = savedInstanceState.getParcelable(STATE_SIGNATURE);
    }

    private void deliverResult() {
        Intent intent = new Intent();

        if (imageView.getDrawable() == null) {
            setResult(RESULT_CANCELED, intent);
            finish();
            return;
        }

        if (signature != null) {
            updateSignatureParams();

            intent.putExtra(RESULT_SIGNATURE, signature);
        }

        setResult(RESULT_OK, intent);
        finish();
    }

    @Override
    protected void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);

        if (signature != null) {
            updateSignatureParams();
        }

        outState.putParcelable(STATE_SIGNATURE, signature);
    }

    private void updateSignatureParams() {
        signature.setOrigin(imageView.getBlockOrigin());
        signature.setScale(
                progressToScale(sizeBar.getProgress()) * (imageView.getDrawable().getIntrinsicWidth() / (float) imageView.getWidth())
        );
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        getMenuInflater().inflate(R.menu.edit_signature_menu, menu);

        addSignature = menu.findItem(R.id.add_signature);
        editSignature = menu.findItem(R.id.edit_signature);
        clearSignature = menu.findItem(R.id.clear_signature);

        addSignature.setOnMenuItemClickListener(new MenuItem.OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(MenuItem menuItem) {
                requestSignature();

                return true;
            }
        });

        clearSignature.setOnMenuItemClickListener(new MenuItem.OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(MenuItem menuItem) {
                resetSignature();

                return true;
            }
        });

        editSignature.setOnMenuItemClickListener(new MenuItem.OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(MenuItem menuItem) {
                requestSignature();

                return true;
            }
        });

        return true;
    }

    private void resetSignature() {
        signature = null;
        updateSignature();
    }

    @Override
    public boolean onPrepareOptionsMenu(Menu menu) {
        final boolean signatureValid = isSignatureValid();

        addSignature.setVisible(!signatureValid);
        editSignature.setVisible(signatureValid);
        clearSignature.setVisible(signatureValid);

        return imageView.getDrawable() != null;
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);

        if (requestCode == CREATE_SIGNATURE_REQUEST_CODE && resultCode == RESULT_OK) {
            signature = data.getParcelableExtra(CreateSignatureActivity.RESULT_SIGNATURE);
            signatureBitmap = null;

            try {
                Bundle args = buildLoaderArgs();
                getSupportLoaderManager().restartLoader(0, args, this);
            } catch (IOException e) {
                DebugLog.logException(e);
            }
        }
    }

    private void updateSignature() {
        final boolean signatureValid = isSignatureValid();

        ViewUtils.postOnPreDraw(imageView, new Runnable() {
            @Override
            public void run() {
                assignSignature();
            }
        });

        final int signatureUiVisibility = signatureValid ? View.VISIBLE : View.GONE;

        findViewById(R.id.bottom_bar_bg).setVisibility(signatureUiVisibility);
        findViewById(R.id.smaller_icon).setVisibility(signatureUiVisibility);
        findViewById(R.id.larger_icon).setVisibility(signatureUiVisibility);

        sizeBar.setVisibility(signatureUiVisibility);

        invalidateOptionsMenu();
    }

    private void assignSignature() {
        final boolean signatureValid = isSignatureValid();

        if (signatureValid) {
            final Drawable drawable = new BitmapDrawable(getResources(), signatureBitmap);

            if (!isSignatureModifiedByUser()) {
                imageView.setBlock(drawable);

                sizeBar.setProgress(DEFAULT_SCALE_PROGRESS);
                updateSignatureParams();
            } else {
                final float scale = signature.getScale() * (imageView.getWidth() / (float) imageView.getDrawable().getIntrinsicWidth());

                imageView.setBlock(drawable, signature.getOrigin(), scale);

                sizeBar.setProgress(scaleToProgress(scale));
                updateSignatureParams();
            }
        } else {
            imageView.setBlock(null);
        }
    }

    private float progressToScale(int progress) {
        return (progress / (float) DEFAULT_SCALE_PROGRESS) * (1f - MIN_SCALE) + MIN_SCALE;
    }

    private int scaleToProgress(float scale) {
        return (int) ((scale - MIN_SCALE) * DEFAULT_SCALE_PROGRESS / (1f - MIN_SCALE));
    }

    private boolean isSignatureValid() {
        return (signature != null && imageView.getDrawable() != null && BitmapUtils.isBitmapValid(signatureBitmap));
    }

    private boolean isSignatureModifiedByUser() {
        return signature.getOrigin().x > 0f && signature.getOrigin().y > 0f;
    }

    @Override
    public Loader<Bitmap> onCreateLoader(int id, Bundle args) {
        final String path = args.getString(LOADER_PATH);
        return new PageBitmapLoader(
                this,
                path,
                bitmapLruCache,
                null,
                false
        );
    }

    @Override
    public void onLoadFinished(Loader<Bitmap> loader, Bitmap data) {
        data = rotateBitmap(data);

        imageView.setImageBitmap(data);

        updateSignature();

        if (signature != null && signatureBitmap == null) {
            new LoadSignatureBitmapTask(signature.getImagePath()).execute();
        }
    }

    private Bitmap rotateBitmap(Bitmap data) {
        if (!BitmapUtils.isBitmapValid(data)) {
            return null;
        }

        Matrix matrix = new Matrix();
        matrix.setRotate(page.getRotationType().getDegrees());

        return Bitmap.createBitmap(data, 0, 0, data.getWidth(), data.getHeight(), matrix, false);
    }

    @Override
    public void onLoaderReset(Loader<Bitmap> loader) {
        imageView.setImageDrawable(null);
    }

    private void requestSignature() {
        startActivityForResult(
                new Intent(this, CreateSignatureActivity.class),
                CREATE_SIGNATURE_REQUEST_CODE
        );
    }

    private Bundle buildLoaderArgs() throws IOException {
        Bundle args = new Bundle();
        args.putString(LOADER_PATH, page.getFilePath(pageStoreStrategy, Page.ImageType.OPTIMIZED_PREVIEW));
        return args;
    }

    /**
     * Loads {@link net.doo.snap.lib.persistence.Signature}'s image from file.
     */
    private class LoadSignatureBitmapTask extends AsyncTask<Void, Void, Bitmap> {

        private final String path;

        private LoadSignatureBitmapTask(String path) {
            this.path = path;
        }

        @Override
        protected Bitmap doInBackground(Void... voids) {
            return BitmapFactory.decodeFile(path);
        }

        @Override
        protected void onPostExecute(Bitmap bitmap) {
            signatureBitmap = bitmap;
            updateSignature();
        }
    }
}
