package net.doo.snap.lib.sensor;

import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.util.FloatMath;

import com.google.inject.Inject;

import java.util.ArrayList;

import roboguice.activity.event.OnPauseEvent;
import roboguice.activity.event.OnResumeEvent;
import roboguice.event.Observes;

/**
 * Helps to determine device movements
 */
public class SensorHelper implements SensorEventListener {

    private SensorManager sensorManager;
    private Sensor accelerometer;

    private float[] gravity;
    private float acceleration;
    private float currentAcceleration;
    private float lastAcceleration;

    private ArrayList<SignificantMoveListener> listeners = new ArrayList<SignificantMoveListener>();

    @Inject
    public SensorHelper(SensorManager sensorManager) {
        this.sensorManager = sensorManager;
        accelerometer = sensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER);
        acceleration = 0.00f;
        currentAcceleration = SensorManager.GRAVITY_EARTH;
        lastAcceleration = SensorManager.GRAVITY_EARTH;
    }

    private void registerListener() {
        if (accelerometer != null) {
            sensorManager.registerListener(this, accelerometer, SensorManager.SENSOR_DELAY_UI);
        }
    }

    private void unregisterListener() {
        sensorManager.unregisterListener(this);
    }

    private void onActivityResume(@Observes OnResumeEvent event) {
        if (!listeners.isEmpty()) {
            registerListener();
        }
    }

    private void onActivityPause(@Observes OnPauseEvent event) {
        unregisterListener();
    }

    /**
     * Registers {@code SignificantMoveListener} to notify about significant moves
     * @param listener
     */
    public void registerSignificantMoveListener(SignificantMoveListener listener) {
        if (listener != null && !listeners.contains(listener)) {
            listeners.add(listener);

            if (listeners.size() == 1) {
                registerListener();
            }
        }
    }

    /**
     * Unregisters {@code SignificantMoveListener}
     * @param listener
     */
    public void unregisterSignificantMoveListener(SignificantMoveListener listener) {
        listeners.remove(listener);

        if (listeners.isEmpty()) {
            unregisterListener();
        }
    }

    @Override
    public void onSensorChanged(SensorEvent event) {
        if (event.sensor.getType() == Sensor.TYPE_ACCELEROMETER){
            gravity = event.values.clone();
            // Shake detection
            float x = gravity[0];
            float y = gravity[1];
            float z = gravity[2];
            lastAcceleration = currentAcceleration;
            currentAcceleration = FloatMath.sqrt(x * x + y * y + z * z);
            float delta = currentAcceleration - lastAcceleration;
            acceleration = acceleration * 0.9f + delta;
            // Make this higher or lower according to how much
            // motion you want to detect
            for (SignificantMoveListener listener : listeners) {
                listener.getSignificantMoveThreshold();
                if(acceleration > listener.getSignificantMoveThreshold()){
                    listener.onSignificantMove();
                }
            }
        }
    }

    @Override
    public void onAccuracyChanged(Sensor sensor, int accuracy) {
        // do nothing
    }
}
