package net.doo.snap.lib.persistence;

import android.content.Context;

import com.google.inject.Inject;

import org.apache.commons.io.FileUtils;

import java.io.File;
import java.io.IOException;

import static net.doo.snap.lib.util.FileUtils.getExternalFilesDirOrShowError;

/**
 * Provides {@link java.io.File}s for images of {@link net.doo.snap.lib.persistence.Signature}s.
 */
public class SignatureStoreStrategy {

    private static final String LAST_DRAWN_SIGNATURE = "lastDrawnSignature";
    private static final String SIGNATURES_FOLDER = "signatures";
    @Inject
    private Context context;

    @Inject
    public SignatureStoreStrategy(Context context) {
        this.context = context;
    }

    /**
     * @return {@link java.io.File} for the image of provided {@link net.doo.snap.lib.persistence.Signature}
     * @throws java.io.IOException if {@link java.io.File} can't be opened
     */
    public File getSignatureFile(Signature signature) throws IOException {
        return getSignatureFile(signature.getId());
    }

    /**
     * @return {@link java.io.File} for the image of provided {@link net.doo.snap.lib.persistence.Signature} id
     * @throws java.io.IOException if {@link java.io.File} can't be opened
     */
    public File getSignatureFile(String id) throws IOException {
        File dir = getExternalFilesDirOrShowError(context, SIGNATURES_FOLDER);

        return FileUtils.getFile(dir, id);
    }

    /**
     * @return {@link java.io.File} of last drawn signature
     * @throws IOException in case if there is no such {@link java.io.File}
     */
    public File getLastDrawnSignatureFile() throws IOException {
        return FileUtils.getFile(
                getExternalFilesDirOrShowError(context, SIGNATURES_FOLDER),
                LAST_DRAWN_SIGNATURE
        );
    }

    /**
     * Removes signature files from filesystem. Method works synchronously - don't call it from UI thread.
     */
    public void eraseSignature(final String signatureId) throws IOException {
        final File signatureFile = getSignatureFile(signatureId);
        if (signatureFile == null) {
            throw new IOException("Can't get file");
        }

        FileUtils.forceDelete(signatureFile);
    }

}
