package net.doo.snap.lib.ui;

import android.app.ActionBar;
import android.content.Intent;
import android.content.SharedPreferences;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.AsyncTask;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;

import com.google.inject.Inject;

import net.doo.snap.lib.R;
import net.doo.snap.lib.analytics.ActivityAnalytics;
import net.doo.snap.lib.persistence.Signature;
import net.doo.snap.lib.persistence.SignatureStoreStrategy;
import net.doo.snap.lib.sensor.SensorHelper;
import net.doo.snap.lib.sensor.ShakeListener;
import net.doo.snap.lib.ui.widget.signature.SignatureView;
import net.doo.snap.lib.util.bitmap.BitmapLruCache;
import net.doo.snap.lib.util.bitmap.BitmapUtils;
import net.doo.snap.lib.util.log.DebugLog;

import org.apache.commons.io.IOUtils;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.UUID;

import roboguice.activity.RoboActivity;

/**
 * Workflow for creating a new signature
 */
public class CreateSignatureActivity extends RoboActivity {

    /**
     * If specified, signature with given id will be updated instead of creating a new one
     */
    public static final String ARG_SIGNATURE_ID = "ARG_SIGNATURE_ID";

    /**
     * {@link net.doo.snap.lib.persistence.Signature} that was created or modified
     */
    public static final String RESULT_SIGNATURE = "RESULT_SIGNATURE";

    /**
     * Id of {@link net.doo.snap.lib.persistence.Signature} that should be replaced with new
     * one
     *
     * @see #RESULT_SIGNATURE
     */
    public static final String RESULT_UPDATED_SIGNATURE_ID = "RESULT_UPDATED_SIGNATURE_ID";

    private static final float RESULT_SCALE = 0.5f;
    private static final float MIN_SIGNATURE_DIMENSION_DP = 38f;
    private static final String SAVED_SIGNATURE_PATH = "SAVED_SIGNATURE_PATH";

    @Inject
    private SignatureStoreStrategy signatureStoreStrategy;
    @Inject
    private ActivityAnalytics activityAnalytics;
    @Inject
    private SensorHelper sensorHelper;
    @Inject
    private SharedPreferences sharedPreferences;
    @Inject
    private BitmapLruCache bitmapLruCache;

    @Nullable
    private String signatureId;

    private SignatureView signatureView;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_create_signature);

        signatureId = getIntent().getStringExtra(ARG_SIGNATURE_ID);

        ActionBar actionBar = getActionBar();
        actionBar.setDisplayShowHomeEnabled(false);
        actionBar.setDisplayShowTitleEnabled(false);
        actionBar.setDisplayShowCustomEnabled(true);
        actionBar.setDisplayHomeAsUpEnabled(false);
        actionBar.setCustomView(R.layout.done);

        View done = findViewById(R.id.done);
        done.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                new CreateSignatureTask(signatureView.getSignature()).execute();
            }
        });

        signatureView = (SignatureView) findViewById(R.id.signature_view);
        sensorHelper.registerSignificantMoveListener(new ShakeListener() {
            @Override
            public void onShake() {
                signatureView.clear();
            }
        });

        if (TextUtils.isEmpty(signatureId) && sharedPreferences.contains(SAVED_SIGNATURE_PATH)) {
            new LoadSavedSignatureTask().execute();
        }
    }

    private void deliverResult(Signature signature) {
        Intent result = new Intent();
        result.putExtra(RESULT_SIGNATURE, signature);

        if (signatureId != null) {
            result.putExtra(RESULT_UPDATED_SIGNATURE_ID, signatureId);
        }

        setResult(RESULT_OK, result);
        finish();
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        getMenuInflater().inflate(R.menu.create_signature_menu, menu);

        return true;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        if (item.getItemId() == R.id.clear_signature) {
            signatureView.clear();

            return true;
        }

        return super.onOptionsItemSelected(item);
    }

    /**
     * Loads previously saved signature {@link android.graphics.Bitmap}
     */
    private class LoadSavedSignatureTask extends AsyncTask<Void, Void, Bitmap> {

        @Override
        protected Bitmap doInBackground(Void... voids) {
            return BitmapFactory.decodeFile(
                    sharedPreferences.getString(SAVED_SIGNATURE_PATH, null)
            );
        }

        @Override
        protected void onPostExecute(Bitmap bitmap) {
            signatureView.setSignature(bitmap);
        }
    }

    /**
     * Creates {@link net.doo.snap.lib.persistence.Signature} image file from provided {@link android.graphics.Bitmap}
     * and assigns a new {@link net.doo.snap.lib.persistence.Signature} generated from it.
     */
    private class CreateSignatureTask extends AsyncTask<Void, Void, Signature> {

        private Bitmap signatureBitmap;
        private DisplayMetrics displayMetrics;

        private CreateSignatureTask(Bitmap signatureBitmap) {
            this.signatureBitmap = signatureBitmap;
            displayMetrics = getResources().getDisplayMetrics();
        }

        @Override
        protected Signature doInBackground(Void... voids) {
            if (signatureBitmap == null) {
                sharedPreferences.edit().putString(SAVED_SIGNATURE_PATH, null).apply();
                return null;
            }

            rememberSignature();
            scaleDownSignature();
            cropSignature();

            return buildSignature(signatureBitmap);
        }

        private void rememberSignature() {
            try {
                File file = signatureStoreStrategy.getLastDrawnSignatureFile();
                BitmapUtils.compress(signatureBitmap, Bitmap.CompressFormat.PNG, 0, file);

                sharedPreferences.edit().putString(SAVED_SIGNATURE_PATH, file.getPath()).apply();
            } catch (IOException e) {
                DebugLog.logException(e);
            }
        }

        private void scaleDownSignature() {
            Matrix matrix = new Matrix();
            matrix.setScale(RESULT_SCALE, RESULT_SCALE);
            signatureBitmap = Bitmap.createBitmap(
                    signatureBitmap,
                    0, 0,
                    signatureBitmap.getWidth(), signatureBitmap.getHeight(),
                    matrix, true
            );
        }

        private void cropSignature() {
            final int minSize = (int) TypedValue.applyDimension(
                    TypedValue.COMPLEX_UNIT_DIP,
                    MIN_SIGNATURE_DIMENSION_DP,
                    displayMetrics
            );

            Rect rect = BitmapUtils.getContentBounds(signatureBitmap);

            Bitmap bitmap = Bitmap.createBitmap(
                    Math.max(rect.width(), minSize),
                    Math.max(rect.height(), minSize),
                    Bitmap.Config.ARGB_8888
            );

            Canvas canvas = new Canvas(bitmap);
            canvas.drawBitmap(
                    signatureBitmap,
                    (bitmap.getWidth() - rect.width()) / 2f - rect.left,
                    (bitmap.getHeight() - rect.height()) / 2f - rect.top,
                    null
            );

            signatureBitmap = bitmap;
        }

        private Signature buildSignature(Bitmap signatureBitmap) {
            if (!BitmapUtils.isBitmapValid(signatureBitmap)) {
                return null;
            }

            Signature signature = new Signature();
            signature.setId(UUID.randomUUID().toString());

            try {
                signature.setImagePath(saveSignatureBitmapToFile(signature, signatureBitmap));
            } catch (IOException e) {
                DebugLog.logException(e);

                return null;
            }

            signature.setRect(new RectF());

            bitmapLruCache.remove(signature.getImagePath());

            return signature;
        }

        private String saveSignatureBitmapToFile(Signature signature, Bitmap signatureBitmap) throws IOException {
            File file = signatureStoreStrategy.getSignatureFile(signature);
            if (file == null) {
                throw new IOException("Can't open file for signature");
            }

            FileOutputStream outputStream = new FileOutputStream(file);
            try {
                signatureBitmap.compress(Bitmap.CompressFormat.PNG, 0, outputStream);
            } finally {
                IOUtils.closeQuietly(outputStream);
            }

            return file.getPath();
        }

        @Override
        protected void onPostExecute(Signature signature) {
            deliverResult(signature);
        }
    }
}
