package net.doo.snap.lib.snap.camera;

import android.animation.ObjectAnimator;
import android.animation.TypeEvaluator;
import android.animation.ValueAnimator;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.Rect;
import android.hardware.Camera;
import android.util.Property;
import android.view.View;

import com.commonsware.cwac.camera.CameraView;

import net.doo.snap.lib.R;
import net.doo.snap.lib.detector.CameraDetectorListener;
import net.doo.snap.lib.detector.DetectionResult;
import net.doo.snap.lib.sensor.SignificantMoveListener;
import net.doo.snap.lib.snap.AutoSnappingDetectionHelper;
import net.doo.snap.lib.util.snap.PolygonHelper;
import net.doo.snap.lib.util.snap.Utils;
import net.doo.snap.lib.util.CameraConfiguration;
import net.doo.snap.lib.util.log.DebugLog;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Implementation of {@link com.commonsware.cwac.camera.CameraView}
 * Draws detected image analysis results
 *
 * @see net.doo.snap.lib.detector.ContourDetector
 * @see DetectionTask
 * @see CameraPreviewFragment
 */
public class SnapCameraView extends CameraView implements CameraDetectorListener, SignificantMoveListener {

    private static final Property<Paint, Integer> PAINT_ALPHA_PROPERTY = Property.of(Paint.class, Integer.TYPE, "alpha");
    private static final int POLYGON_FADE_OUT_DURATION_MS = 250;
    private static final int DELAY_AFTER_MOVE_MS = 2000;

    private static final int MIN_AREA_COORD = -1000;
    private static final int MAX_AREA_COORD = 1000;

    private static final int AREA_WEIGHT = 1000;

    private List<PointF> polygon;
    private final Paint paint;
    private boolean isAutoSnapEnabled = true;
    private boolean isAutoFocusing = false;

    private ValueAnimator polygonAnimator;
    private ObjectAnimator polygonAlphaAnimator;
    private SnapCameraView.PolygonEvaluator polygonEvaluator = new PolygonEvaluator();

    private AutoSnappingDetectionHelper detectionHelper;
    private CameraConfiguration cameraConfiguration;

    private boolean cameraAvailable = false;

    private long lastSignificantMove;

    /**
     * points to be drawn on canvas for better performance
     */
    private float[] points;
    private float[] animationPoints;

    private PolygonHelper polygonHelper;
    private boolean drawPolygon = true;

    private byte[] previewBuffer;

    public SnapCameraView(Context context, AutoSnappingDetectionHelper detectionHelper, CameraConfiguration cameraConfiguration) {
        super(context);

        this.detectionHelper = detectionHelper;
        this.cameraConfiguration = cameraConfiguration;

        paint = new Paint();
        paint.setColor(getResources().getColor(R.color.polygon_lines));
        paint.setStyle(Paint.Style.STROKE);
        paint.setStrokeWidth(getResources().getDimension(R.dimen.polygon_stroke_width));
        paint.setAntiAlias(true);

        points = new float[16];
        animationPoints = new float[points.length];

        polygonHelper = new PolygonHelper();
        polygon = Collections.emptyList();
    }

    /**
     * @return polygon containing document inside snapped image
     */
    public List<PointF> getPolygon() {
        return polygon;
    }

    /**
     * Sets option to snap image as soon as good polygon is detected
     *
     * @param enabled
     */
    public void setAutoSnapEnabled(boolean enabled) {
        isAutoSnapEnabled = enabled;
        if (!isAutoSnapEnabled) {
            detectionHelper.reset(true);
        }
    }

    @Override
    protected void dispatchDraw(Canvas canvas) {
        super.dispatchDraw(canvas);

        if (drawPolygon && !polygon.isEmpty()) {
            canvas.drawLines(points, paint);
        }
    }

    @Override
    public void onCameraOpen(Camera camera) throws RuntimeException {
        super.onCameraOpen(camera);
        if(camera != null) {
            Camera.Parameters parameters = camera.getParameters();
            if (parameters != null && !cameraConfiguration.isConigured()) {
                cameraConfiguration.loadCameraParameters(parameters);
            }
        }
    }

    @Override
    protected void onLayout(boolean changed, int l, int t, int r, int b) {
        super.onLayout(changed, l, t, r, b);
        View view = getChildAt(0);
        if (view != null) {
            polygonHelper.setLayout(view.getLeft(), view.getTop(), view.getRight(), view.getBottom());
        }
    }

    @Override
    public void autoFocus() {
        isAutoFocusing = true;

        try {
            Camera.Parameters params = getCameraParameters();
            Utils.enableAutoFocus(params);
            setFocusAndMeteringArea(params);
            setCameraParameters(params);
            super.autoFocus();
        } catch (RuntimeException e) {
            DebugLog.logException(e);
        }
    }

    /**
     * Sets area to focus. Based on current polygon
     * @param parameters
     */
    private void setFocusAndMeteringArea(Camera.Parameters parameters) {
        List<Camera.Area> areas = null;

        if (parameters.getMaxNumFocusAreas() > 0) {
            areas = getAreas();
            parameters.setFocusAreas(areas);
        }

        if (parameters.getMaxNumMeteringAreas() > 0) {
            if (areas == null) {
                areas = getAreas();
            }
            parameters.setMeteringAreas(areas);
        }
    }

    private List<Camera.Area> getAreas() {
        List<Camera.Area> areas = null;
        if (!polygon.isEmpty()) {
            float minX = 1;
            float maxX = 0;
            float minY = 1;
            float maxY = 0;

            for (PointF point : polygon) {
                if (point.x < minX) minX = point.x;
                if (point.x > maxX) maxX = point.x;
                if (point.y < minY) minY = point.y;
                if (point.y > maxY) maxY = point.y;
            }

            Rect areaRect = new Rect(
                    polygonCoordToAreaCoord(minX),
                    polygonCoordToAreaCoord(minY),
                    polygonCoordToAreaCoord(maxX),
                    polygonCoordToAreaCoord(maxY));

            // weight can be from 1 to 1000, let's set maximum
            Camera.Area area = new Camera.Area(areaRect, AREA_WEIGHT);
            areas = new ArrayList<Camera.Area>();
            areas.add(area);
        }

        return areas;
    }

    /**
     * Transforms 0..1 float coordinate to integer -1000..1000 coordinate
     * @param coordinate
     * @return
     */
    private int polygonCoordToAreaCoord(float coordinate) {
        int result = Math.round(coordinate * MAX_AREA_COORD * 2 + MIN_AREA_COORD);
        if (result > MAX_AREA_COORD) result = MAX_AREA_COORD;
        if (result < MIN_AREA_COORD) result = MIN_AREA_COORD;
        return result;
    }

    @Override
    public void onAutoFocus(boolean success, Camera camera) {
        /**
         * Using isAutoFocusing flag to ensure we called {@link #autoFocus()}
         * before getting this callback
         */
        if (isAutoFocusing) {
            isAutoFocusing = false;
            super.onAutoFocus(success, camera);
        }
    }

    @Override
    public void onDetectionOK(final List<PointF> polygon) {
        if (!this.polygon.isEmpty()) {
            animateToPolygon(polygon);
        } else {
            polygonHelper.polygonToPoints(polygon, points);
            invalidate();
        }

        this.polygon = polygon;

        boolean movedRecently = System.currentTimeMillis() - lastSignificantMove <= DELAY_AFTER_MOVE_MS;

        SnapCameraHost host = (SnapCameraHost) getHost();
        if (!movedRecently && isAutoSnapEnabled && host.useContourDetection()) {
            detectionHelper.onResult(DetectionResult.OK);

            if (!isAutoFocusing) {
                autoFocus();
            }
        }
        addPreviewCallbackBuffer(previewBuffer);
    }

    @Override
    public void onBarcodeDetectionOK(String content) {
        //continue
    }

    @Override
    public void onDetectionWithError(DetectionResult result, List<PointF> polygon) {
        if (isAutoSnapEnabled) {
            detectionHelper.onResult(result);
        }

        this.polygon = polygon;
        animateToPolygon(polygon);

        addPreviewCallbackBuffer(previewBuffer);
    }

    @Override
    public void onDetectionFailed(DetectionResult result) {
        if (isAutoSnapEnabled) {
            detectionHelper.onResult(result);
        }

        animateToPolygon(Collections.<PointF>emptyList());

        addPreviewCallbackBuffer(previewBuffer);
    }

    private void animateToPolygon(List<PointF> polygon) {
        if (polygonAnimator != null) {
            polygonAnimator.cancel();
            polygonAnimator = null;
        }

        if (polygonAlphaAnimator != null) {
            polygonAlphaAnimator.cancel();
            polygonAlphaAnimator = null;
        }

        if (polygon.isEmpty()) {
            animatePolygonFadeOut();
            return;
        }

        paint.setAlpha(255);

        polygonHelper.polygonToPoints(polygon, animationPoints);

        polygonAnimator = ValueAnimator.ofObject(
                polygonEvaluator,
                points, animationPoints
        );
        polygonAnimator.start();
    }

    private void animatePolygonFadeOut() {
        polygonAlphaAnimator = ObjectAnimator.ofInt(paint, PAINT_ALPHA_PROPERTY, 0);

        polygonAlphaAnimator.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
            @Override
            public void onAnimationUpdate(ValueAnimator valueAnimator) {
                invalidate();
            }
        });

        polygonAlphaAnimator.setDuration(POLYGON_FADE_OUT_DURATION_MS);

        polygonAlphaAnimator.start();
    }

    /**
     * Hides all notifications that help while auto snapping
     * Dismisses toast, hides "Move closer" arrows and "Rotate" image
     */
    public void resetNotifications() {
        detectionHelper.reset(true);
    }

    @Override
    public void onResume() {
        super.onResume();
        cameraAvailable = true;
    }

    @Override
    public void onPause() {
        super.onPause();
        cameraAvailable = false;
    }

    @Override
    public void startPreview() {
        if (cameraAvailable) {
            super.startPreview();
            polygon = Collections.emptyList();
            Camera.Parameters parameters = getCameraParameters();
            setFocusAndMeteringArea(parameters);
            setCameraParameters(parameters);

            previewBuffer = ((SnapCameraHost) getHost()).getNewPreviewBuffer();
            addPreviewCallbackBuffer(previewBuffer);

            polygonHelper.setRotation(getDisplayOrientation());
        }
    }

    @Override
    public void stopPreview() {
        if (cameraAvailable) {
            super.stopPreview();
        }
    }

    /**
     * Indicates whether polygon should be drawn or not.
     *
     * @param drawPolygon
     */
    public void setDrawPolygon(boolean drawPolygon) {
        if (drawPolygon && !this.drawPolygon) {
            addPreviewCallbackBuffer(previewBuffer);
        }
        this.drawPolygon = drawPolygon;
        invalidate();
    }

    @Override
    public float getSignificantMoveThreshold() {
        return SignificantMoveListener.THRESHOLD_LOW;
    }

    @Override
    public void onSignificantMove() {
        lastSignificantMove = System.currentTimeMillis();
    }

    /**
     * Performs linear translation of arbitrary float array values to polygon drawing points
     */
    private class PolygonEvaluator implements TypeEvaluator {

        @Override
        public Object evaluate(float fraction, Object startObj, Object endObj) {
            evaluatePolygon(
                    fraction,
                    (float[]) startObj,
                    (float[]) endObj
            );

            return null;
        }

        private void evaluatePolygon(float fraction, float[] start, float[] end) {
            for (int i = 0; i < points.length; i++) {
                points[i] = start[i] + fraction * (end[i] - start[i]);
            }

            invalidate();
        }
    }
}
