package net.doo.snap.lib.smartname;

import android.content.SharedPreferences;
import com.google.inject.Inject;
import net.doo.snap.lib.PreferencesConstants;
import net.doo.snap.lib.util.smartname.SmartNameUtil;
import org.apache.commons.lang.StringUtils;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Generates document name
 */
public class SmartNameGenerator {

    public static final String SEPARATOR = "|";
    public static final String TERM_CHARACTER = "\t";
    public static final String DEFAULT_TEMPLATE = "Scan "
            + TERM_CHARACTER
            + "-"
            + TERM_CHARACTER
            + "-"
            + TERM_CHARACTER
            + " "
            + TERM_CHARACTER
            + "."
            + TERM_CHARACTER
            + "."
            + TERM_CHARACTER;

    public static final List<TermType> DEFAULT_TERMS = Arrays.asList(
            TermType.YEAR,
            TermType.MONTH,
            TermType.DAY,
            TermType.HOURS,
            TermType.MINUTES,
            TermType.SECONDS);

    private static final Pattern PATTERN = Pattern.compile(TERM_CHARACTER);

    @Inject
    private SharedPreferences preferences;
    @Inject
    private SmartNameValuesProvider smartNameValuesProvider;

    /**
     * Generates document name
     * @param template  smart name template
     * @param terms     {@link net.doo.snap.lib.smartname.TermType}s for template variables
     * @return document name
     */
    public String generateDocumentName(String template, List<TermType> terms) {
        Matcher matcher = PATTERN.matcher(template);

        StringBuffer resultBuilder = new StringBuffer();
        int index = 0;
        while(matcher.find()) {
            matcher.appendReplacement(resultBuilder, smartNameValuesProvider.getTermValue(terms.get(index++)));
        }
        matcher.appendTail(resultBuilder);
        return resultBuilder.toString();
    }

    /**
     * Generates document name with template saved in shared preferences
     * @return document name
     */
    public String generateDocumentName() {
        String template = preferences.getString(PreferencesConstants.SMART_NAME_TEMPLATE, null);
        String termsTypes = preferences.getString(PreferencesConstants.SMART_NAME_TERMS_TYPES, null);
        List<TermType> terms = new ArrayList<>();
        if (StringUtils.isEmpty(template)) {
            template = DEFAULT_TEMPLATE;
            terms.addAll(DEFAULT_TERMS);
        } else {
            if (StringUtils.isEmpty(termsTypes)) {
                return template;
            }
            terms.addAll(SmartNameUtil.convertTermsFromPreferencesFormat(termsTypes));
        }

        return generateDocumentName(template, terms);
    }

}
