package net.doo.snap.lib.ui.widget;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.util.AttributeSet;
import android.util.SparseIntArray;
import android.view.GestureDetector;
import android.view.MotionEvent;
import android.widget.ImageButton;

import java.util.ArrayList;

/**
 * {@link android.widget.ImageButton} which circulates through list of defined states on each click.
 * Added states are not saved between configuration changes.
 */
public class MultiStateImageButton extends ImageButton {

    /**
     * Listens for state changes of {@link net.doo.snap.lib.ui.widget.MultiStateImageButton}
     */
    public interface OnStateChangedListener {

        /**
         * Invoked when state of {@link net.doo.snap.lib.ui.widget.MultiStateImageButton} is about to change.
         * Gives ability to cancel state change, but returning {@code false} value.
         *
         * @return {@code true} if state should be changed. {@code false} otherwise.
         */
        boolean onStateChange(int newState);

    }

    private GestureDetector gestureDetector;

    private ArrayList<State> states = new ArrayList<State>();
    private SparseIntArray stateIdToPosition = new SparseIntArray();
    private int currentStateIndex = -1;

    private OnStateChangedListener onStateChangedListener;

    public MultiStateImageButton(Context context) {
        super(context);

        initView(context);
    }

    public MultiStateImageButton(Context context, AttributeSet attrs) {
        super(context, attrs);

        initView(context);
    }

    public MultiStateImageButton(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);

        initView(context);
    }

    private void initView(Context context) {
        gestureDetector = new GestureDetector(context, new GestureListener());
    }

    @Override
    public void setOnClickListener(OnClickListener l) throws UnsupportedOperationException {
        throw new UnsupportedOperationException();
    }

    /**
     * Assigns {@link net.doo.snap.lib.ui.widget.MultiStateImageButton.OnStateChangedListener} which will be
     * notified each time state is changed
     */
    public void setOnStateChangedListener(OnStateChangedListener onStateChangedListener) {
        this.onStateChangedListener = onStateChangedListener;
    }

    /**
     * @param id          integer identifier of the state
     * @param drawableRes {@link android.graphics.drawable.Drawable} resource which will be applied to button in provided state
     * @return same {@link net.doo.snap.lib.ui.widget.MultiStateImageButton}
     */
    public MultiStateImageButton addState(int id, int drawableRes) {
        return this.addState(id, getResources().getDrawable(drawableRes));
    }

    /**
     * @param id       integer identifier of the state
     * @param drawable {@link android.graphics.drawable.Drawable} which will be applied to button in provided state
     * @return same {@link net.doo.snap.lib.ui.widget.MultiStateImageButton}
     */
    public MultiStateImageButton addState(int id, Drawable drawable) {
        State state = new State(id, drawable);
        stateIdToPosition.put(id, states.size());
        states.add(state);

        if (currentStateIndex < 0) {
            currentStateIndex = 0;
            updateDrawable();
        }

        return this;
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        super.onTouchEvent(event);

        return gestureDetector.onTouchEvent(event);
    }

    /**
     * Moves {@link net.doo.snap.lib.ui.widget.MultiStateImageButton} to the next state and notifies
     * {@link net.doo.snap.lib.ui.widget.MultiStateImageButton.OnStateChangedListener} (if it was set).
     */
    public void moveToNextState() {
        if (states.isEmpty()) {
            return;
        }

        final int newState = (currentStateIndex + 1) % states.size();
        if (pollListener(newState)) {
            currentStateIndex = newState;
            updateDrawable();
        }
    }

    /**
     * Moves {@link net.doo.snap.lib.ui.widget.MultiStateImageButton} to state with provided id.
     * Note, that {@link net.doo.snap.lib.ui.widget.MultiStateImageButton.OnStateChangedListener} won't
     * be invoked.
     */
    public void setCurrentState(int id) {
        currentStateIndex = getStatePosition(id);

        updateDrawable();
    }

    private int getStatePosition(int id) {
        int position = stateIdToPosition.get(id, -1);
        if (position < 0) {
            throw new IllegalStateException("No state with id = " + id);
        }

        return position;
    }

    private void updateDrawable() {
        setImageDrawable(states.get(currentStateIndex).drawable);
    }

    private boolean pollListener(int newState) {
        return onStateChangedListener == null || onStateChangedListener.onStateChange(states.get(newState).id);
    }

    /**
     * Listens for {@link #onSingleTapUp(android.view.MotionEvent)} event
     */
    private class GestureListener extends GestureDetector.SimpleOnGestureListener {

        @Override
        public boolean onDown(MotionEvent e) {
            return true;
        }

        @Override
        public boolean onSingleTapUp(MotionEvent e) {
            moveToNextState();

            return true;
        }

    }

    /**
     * Represents state of {@link net.doo.snap.lib.ui.widget.MultiStateImageButton} with unique
     * identifier and associated {@link android.graphics.drawable.Drawable}
     */
    private static class State {

        public final int id;
        public final Drawable drawable;

        private State(int id, Drawable drawable) {
            this.id = id;
            this.drawable = drawable;
        }

    }

}
