package net.doo.snap.lib.util.bitmap;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.FileObserver;
import android.os.Handler;
import android.support.v4.content.AsyncTaskLoader;

/**
 * Asynchronously loads {@link android.graphics.Bitmap} from file and caches the
 * result
 */
public class PageBitmapLoader extends AsyncTaskLoader<Bitmap> {

    private static final long INITIAL_RETRY_INTERVAL_MS = 2000L;
    private static final int MAX_RETRY_ATTEMPTS = 8;

    private BitmapLruCache bitmapLruCache;
    protected String imagePath;
    private FileObserver fileObserver;

    private Handler handler = new Handler();
    private Runnable forceLoadRunnable = new Runnable() {
        @Override
        public void run() {
            if (isStarted()) {
                forceLoad();
            }
        }
    };
    private int retryAttempts = 0;

    private Bitmap bitmap;

    public PageBitmapLoader(Context context, String imagePath, BitmapLruCache bitmapLruCache, Bitmap preservedBitmap) {
        this(context, imagePath, bitmapLruCache, preservedBitmap, true);
    }

    public PageBitmapLoader(Context context, String imagePath, BitmapLruCache bitmapLruCache, Bitmap preservedBitmap, boolean registerObserver) {
        super(context);
        this.imagePath = imagePath;
        this.bitmapLruCache = bitmapLruCache;
        this.bitmap = preservedBitmap;

        if (registerObserver) {
            fileObserver = new FileObserver(imagePath, FileObserver.CLOSE_WRITE) {
                @Override
                public void onEvent(int event, String path) {
                    handler.post(forceLoadRunnable);
                }
            };
        } else {
            fileObserver = NULL_FILE_OBSERVER;
        }
    }

    @Override
    public Bitmap loadInBackground() {
        return (imagePath != null)
                ? BitmapFactory.decodeFile(imagePath)
                : null;
    }

    @Override
    public void deliverResult(Bitmap data) {
        handler.removeCallbacks(forceLoadRunnable);

        if (isReset() || data == null) {
            bitmap = null;

            if (!isReset() && retryAttempts < MAX_RETRY_ATTEMPTS) {
                retryAttempts++;
                handler.postDelayed(forceLoadRunnable, INITIAL_RETRY_INTERVAL_MS * retryAttempts);
            }

            return;
        }

        if (bitmapLruCache != null && !data.equals(bitmap)) {
            bitmapLruCache.put(imagePath, data);
        }
        bitmap = data;

        if (isStarted()) {
            fileObserver.startWatching();
            super.deliverResult(bitmap);
        }
    }

    @Override
    protected void onStartLoading() {
        if (imagePath == null) {
            deliverResult(null);
            return;
        }

        if (bitmapLruCache == null) {
            forceLoad();
            return;
        }

        checkCache();

        boolean dataValid = bitmap != null;
        if (dataValid) {
            deliverResult(bitmap);
        }
        if (takeContentChanged() || !dataValid) {
            forceLoad();
        }
    }

    @Override
    protected void onReset() {
        super.onReset();

        cancelLoad();

        bitmap = null;
        retryAttempts = 0;
        fileObserver.stopWatching();
    }

    @Override
    protected void onAbandon() {
        super.onAbandon();
        fileObserver.stopWatching();
    }

    private void checkCache() {
        if (bitmap == null) {
            bitmap = bitmapLruCache.get(imagePath);
        }
    }

    private static final FileObserver NULL_FILE_OBSERVER = new FileObserver(null) {

        @Override
        public void onEvent(int i, String s) {
        }

        @Override
        protected void finalize() {
        }

        @Override
        public void startWatching() {
        }

        @Override
        public void stopWatching() {
        }

    };

}
