package net.doo.snap.lib.billing;

import android.util.Base64;
import net.doo.snap.lib.genuineness.GenuinenessChecker;

import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.util.UUID;

/**
 * Provides security methods for communication with billing services
 */
public final class SecurityManager {

    public static final String KEY_FACTORY_ALGORITHM = "RSA";
    public static final String SIGNATURE_ALGORITHM = "SHA1withRSA";

    /**
     * @return application {@link java.security.PublicKey} for verification of signatures
     */
    public static PublicKey getPublicKey() throws NoSuchAlgorithmException, InvalidKeySpecException {
        byte[] key = Base64.decode(GenuinenessChecker.BASE64_PUBLIC_KEY, Base64.DEFAULT);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_FACTORY_ALGORITHM);
        return keyFactory.generatePublic(new X509EncodedKeySpec(key));
    }

    /**
     * Verifies signature of message or throws exception on failure.
     *
     * @param message   message which signature has to be verified
     * @param signature signaure of provided message
     * @throws VerificationFailedException in case if signature verification fails for any reason
     */
    public void verifySignature(String message, String signature) throws VerificationFailedException {
        if (message == null || signature == null) {
            throw new VerificationFailedException();
        }

        Signature signatureAlgorithm = buildSignatureAlgorithm();

        verifySignature(
                signatureAlgorithm,
                message.getBytes(),
                Base64.decode(signature, Base64.DEFAULT)
        );
    }

    private void verifySignature(Signature signatureAlgorithm, byte[] messageBytes, byte[] signatureBytes) throws VerificationFailedException {
        try {
            signatureAlgorithm.update(messageBytes);

            if (!signatureAlgorithm.verify(signatureBytes)) {
                throw new VerificationFailedException();
            }
        } catch (SignatureException e) {
            throw new VerificationFailedException(e);
        }
    }

    private Signature buildSignatureAlgorithm() throws VerificationFailedException {
        try {
            Signature signatureAlgorithm = Signature.getInstance(SIGNATURE_ALGORITHM);
            signatureAlgorithm.initVerify(getPublicKey());

            return signatureAlgorithm;
        } catch (NoSuchAlgorithmException e) {
            throw new VerificationFailedException(e);
        } catch (InvalidKeyException e) {
            throw new VerificationFailedException(e);
        } catch (InvalidKeySpecException e) {
            throw new VerificationFailedException(e);
        }
    }

    /**
     * @return pseudo-unique {@link String} value
     */
    public String generateNonce() {
        return UUID.randomUUID().toString();
    }

    /**
     * Thrown when verification of message fails
     */
    public static class VerificationFailedException extends Exception {

        public VerificationFailedException(Throwable throwable) {
            super(throwable);
        }

        public VerificationFailedException() {
        }

    }

}
