package net.doo.snap.lib.persistence;

import android.util.SparseArray;

/**
 * Supported rotation degrees for image editing
 */
public enum RotationType {
    ROTATION_0(0),
    ROTATION_90(90),
    ROTATION_180(180),
    ROTATION_270(270),
    /**
     * Used for animations, since animation framework behaves differently for 0 and 360 degrees
     */
    ROTATION_360(360);

    private final int degrees;

    private static final SparseArray<RotationType> degrees2Type = new SparseArray<RotationType>();

    static {
        for (RotationType type : values()) {
            degrees2Type.put(type.degrees, type);
        }

    }

    RotationType(int degrees) {
        this.degrees = degrees;
    }

    /**
     * @param type
     * @return {@link net.doo.snap.lib.persistence.RotationType} which is result of clockwise rotation
     * of provided {@link net.doo.snap.lib.persistence.RotationType}
     */
    public static RotationType rotateClockwise(RotationType type) {
        RotationType result = ROTATION_0;
        switch (type) {
            case ROTATION_360:
            case ROTATION_0:
                result = ROTATION_90;
                break;
            case ROTATION_90:
                result = ROTATION_180;
                break;
            case ROTATION_180:
                result = ROTATION_270;
                break;
            case ROTATION_270:
                result = ROTATION_360;
                break;
        }
        return result;
    }

    /**
     * @return degrees representation of {@link net.doo.snap.lib.persistence.RotationType}
     */
    public int getDegrees() {
        return degrees;
    }

    /**
     * @param degrees
     * @return {@link net.doo.snap.lib.persistence.RotationType} corresponding to given degrees value
     * @throws java.lang.IllegalStateException in case there is no {@link net.doo.snap.lib.persistence.RotationType} for the given degrees value
     */
    public static RotationType getByDegrees(int degrees) {
        RotationType type = degrees2Type.get(degrees);
        if (type == null) {
            throw new IllegalStateException("no rottation type for: " + degrees);
        }

        return type;
    }
}
