package net.doo.snap.lib.snap.edit.util;

import android.animation.Animator;
import android.view.View;
import android.view.ViewPropertyAnimator;
import android.view.animation.AccelerateDecelerateInterpolator;
import android.view.animation.DecelerateInterpolator;
import android.view.animation.Interpolator;

import net.doo.snap.lib.persistence.Page;
import net.doo.snap.lib.snap.preview.ImagePreviewFragment;
import net.doo.snap.lib.snap.PreviewFragmentPagerAdapter;
import net.doo.snap.lib.util.ui.ViewUtils;
import net.doo.snap.lib.ui.widget.ViewPager;

/**
 * Provides common methods for page modifying
 */
public class PageEditorHelper {

    private static final Interpolator PAGE_SLIDE_IN_INTERPOLATOR = new DecelerateInterpolator();
    private static final Interpolator PAGE_SLIDE_OUT_INTERPOLATOR = new AccelerateDecelerateInterpolator();

    private ViewPager pager;
    private PreviewFragmentPagerAdapter adapter;

    public PageEditorHelper(ViewPager pager, PreviewFragmentPagerAdapter adapter) {
        this.pager = pager;
        this.adapter = adapter;
    }

    /**
     * Slides neighbor page in and delete current page
     *
     * @param listener which will be invoked during animation
     */
    public void deleteCurrentPage(Animator.AnimatorListener listener) {
        if (pager.getCurrentItem() == 0) {
            slideNextPageIn(listener);
        } else {
            slidePreviousPageIn(listener);
        }
    }

    private void slideNextPageIn(Animator.AnimatorListener listener) {
        final int nextPosition = pager.getCurrentItem() + 1;
        if (!isValidPagePosition(nextPosition)) {
            return;
        }

        final View nextView = adapter.getFragment(pager, nextPosition).getView();
        getPageSlideInAndDeleteAnimator(pager.getCurrentItem(), nextView, listener)
                .translationX(-(nextView.getWidth() + pager.getPageMargin()))
                .start();
    }

    private void slidePreviousPageIn(Animator.AnimatorListener listener) {
        final int previousPosition = pager.getCurrentItem() - 1;
        if (!isValidPagePosition(previousPosition)) {
            return;
        }

        final View previousView = adapter.getFragment(pager, previousPosition).getView();
        getPageSlideInAndDeleteAnimator(pager.getCurrentItem(), previousView, listener)
                .translationX(previousView.getWidth() + pager.getPageMargin())
                .start();
    }

    private ViewPropertyAnimator getPageSlideInAndDeleteAnimator(final int positionToDelete, final View viewToSlide, final Animator.AnimatorListener listener) {
        return viewToSlide.animate()
                .setInterpolator(PAGE_SLIDE_IN_INTERPOLATOR)
                .setListener(new ViewUtils.DefaultAnimationListener() {

                    @Override
                    public void onAnimationStart(Animator animation) {
                        if (listener != null) {
                            listener.onAnimationStart(animation);
                        }
                    }

                    @Override
                    public void onAnimationCancel(Animator animation) {
                        if (listener != null) {
                            listener.onAnimationCancel(animation);
                        }
                    }

                    @Override
                    public void onAnimationRepeat(Animator animation) {
                        if (listener != null) {
                            listener.onAnimationRepeat(animation);
                        }
                    }

                    @Override
                    public void onAnimationFinished(Animator animation) {
                        if (!pager.isAttachedToWindow() || !isValidPagePosition(positionToDelete)) {
                            viewToSlide.setTranslationX(ViewUtils.TRANSLATION_DEFAULT);
                            return;
                        }

                        final int currentPosition = pager.getCurrentItem();
                        if (currentPosition == positionToDelete) {
                            pager.setCurrentItem(currentPosition - 1, false);
                        }
                        adapter.deletePage(positionToDelete);
                        viewToSlide.setTranslationX(ViewUtils.TRANSLATION_DEFAULT);

                        if (listener != null) {
                            listener.onAnimationEnd(animation);
                        }
                    }

                });
    }

    /**
     * Inserts page to {@link net.doo.snap.lib.ui.widget.ViewPager} with animation
     *
     * @param page     {@link net.doo.snap.lib.persistence.Page} to insert
     * @param position in adapter at which to insert page
     * @param listener which will be invoked during animation
     */
    public void addPageToPosition(Page page, int position, Animator.AnimatorListener listener) {
        page.getParameters().putBoolean(ImagePreviewFragment.PLAY_DROP_IN_ANIMATION, true);
        page.getParameters().putFloat(ImagePreviewFragment.DROP_X, pager.getWidth() / 2);
        page.getParameters().putFloat(ImagePreviewFragment.DROP_Y, pager.getHeight());

        adapter.addPage(position, page);
        pager.setCurrentItem(position);

        if (adapter.getPagesCount() > 1) {
            if (position == 0) {
                slideNextPageOut(listener);
            } else {
                slidePreviousPageOut(listener);
            }
        }
    }

    /**
     * Slides current page out and adds new page into current position
     *
     * @param page     {@link net.doo.snap.lib.persistence.Page} to add
     * @param listener which will be invoked during animation
     */
    public void addPageToTheLeft(Page page, Animator.AnimatorListener listener) {
        final int currentItem = pager.getCurrentItem();
        adapter.addPage(currentItem, page);

        pager.setCurrentItem(currentItem, false);
        slideNextPageOut(listener);
    }

    /**
     * Slides current page out and adds new page after current position
     *
     * @param page     {@link net.doo.snap.lib.persistence.Page} to add
     * @param listener which will be invoked during animation
     */
    public void addPageToTheRight(Page page, Animator.AnimatorListener listener) {
        final int currentItem = pager.getCurrentItem();
        adapter.addPage(currentItem + 1, page);

        pager.setCurrentItem(currentItem + 1, false);
        slidePreviousPageOut(listener);
    }

    private void slideNextPageOut(Animator.AnimatorListener listener) {
        View view = adapter.getFragment(pager, pager.getCurrentItem() + 1).getView();
        view.setTranslationX(-(view.getWidth() + pager.getPageMargin()));
        getPageSlideOutAnimator(view)
                .setListener(listener)
                .start();
    }

    private void slidePreviousPageOut(Animator.AnimatorListener listener) {
        View view = adapter.getFragment(pager, pager.getCurrentItem() - 1).getView();
        view.setTranslationX(view.getWidth() + pager.getPageMargin());
        getPageSlideOutAnimator(view)
                .setListener(listener)
                .start();
    }

    private ViewPropertyAnimator getPageSlideOutAnimator(final View viewToSlide) {
        return viewToSlide.animate()
                .translationX(ViewUtils.TRANSLATION_DEFAULT)
                .setInterpolator(PAGE_SLIDE_OUT_INTERPOLATOR);
    }

    private boolean isValidPagePosition(int position) {
        return position >= 0
                && position < adapter.getCount() - 1
                && adapter.getFragment(pager, position) != null
                && adapter.getFragment(pager, position).getView() != null;
    }

}
