package net.doo.snap.lib.sensor;

/**
 * Shake listener accumulates several moves that passed threshold
 * and calls abstract {@code onShake()} method
 */
public abstract class ShakeListener implements SignificantMoveListener {
    /**
     * If current move is later than this value in milliseconds than moves counter is reset
     */
    private static final int MAX_SHAKE_TIME_MS = 2000;
    /**
     * Next shake will be not earlier than this value in milliseconds
     */
    private static final int NEXT_SHAKE_DELAY_MS = 3000;
    /**
     * Determines how many moves are needed to call {@code onShake()}
     */
    private static final int SHAKE_MOVES_COUNT = 6;

    private int movesCount;
    private long lastMove;
    private long lastShake;

    @Override
    public float getSignificantMoveThreshold() {
        return SignificantMoveListener.THRESHOLD_HIGH;
    }

    @Override
    public void onSignificantMove() {
        long currentMove = System.currentTimeMillis();
        if (currentMove - lastMove > MAX_SHAKE_TIME_MS) {
            movesCount = 1;
        } else {
            movesCount++;
        }

        if (movesCount >= SHAKE_MOVES_COUNT && currentMove - lastShake > NEXT_SHAKE_DELAY_MS) {
            movesCount = 0;
            lastShake = currentMove;
            onShake();
        }

        lastMove = currentMove;
    }

    /**
     * implement this method to get notified about shake move
     */
    public abstract void onShake();
}
