package net.doo.snap.lib.billing;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Handler;
import android.os.Looper;

import com.google.inject.Inject;

import net.doo.snap.lib.BuildConfig;
import net.doo.snap.lib.PreferencesConstants;
import net.doo.snap.lib.util.billing.IabException;
import net.doo.snap.lib.util.billing.IabHelper;
import net.doo.snap.lib.util.billing.IabResult;
import net.doo.snap.lib.util.billing.Inventory;
import net.doo.snap.lib.util.billing.Purchase;
import net.doo.snap.lib.util.log.DebugLog;

import java.util.Arrays;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * {@link net.doo.snap.lib.billing.BillingManager} implementation
 */
public class BillingManagerImpl implements BillingManager {

    private static final int PURCHASE_REQUEST_CODE = 999000;

    private static final int ITEM_PURCHASED = 0;
    private static final Handler MAIN_LOOPER_HANDLER = new Handler(Looper.getMainLooper());

    private IabHelper iabHelper;

    private ExecutorService singleThreadExecutor;

    @Inject
    private SharedPreferences preferences;

    @Inject
    public BillingManagerImpl() {
    }

    @Override
    public void initialize(Context context, final IabHelper.OnIabSetupFinishedListener listener) {
        singleThreadExecutor = Executors.newSingleThreadExecutor();
        iabHelper = new IabHelper(context);

        iabHelper.startSetup(new IabHelper.OnIabSetupFinishedListener() {
            @Override
            public void onIabSetupFinished(final IabResult result) {
                if (!result.isSuccess()) {
                    DebugLog.e("Problem setting up in-app billing: " + result);
                    notifySetupFinished(listener, result);
                    return;
                }

                // Have we been disposed of in the meantime? If so, quit.
                if (iabHelper == null) return;

                // IAB is fully set up.
                DebugLog.d("Setup successful.");
                refreshItemsAsync(new RefreshItemsListener() {
                    @Override
                    public void onItemsRefreshFinished() {
                        notifySetupFinished(listener, result);
                    }
                });
            }
        });
    }

    @Override
    public boolean isInitialized() {
        return iabHelper != null && iabHelper.isSetupDone();
    }

    @Override
    public boolean isItemAvailable(String itemId) {
        return BuildConfig.DEBUG || preferences.getBoolean(itemId, false);
    }

    @Override
    public boolean isBillingContentAvailable(BillingContent billingContent) {
        return BillingContent.isConentAvailable(billingContent, this);
    }

    @Override
    public String getItemPrice(String itemId) {
        return preferences.getString(PreferencesConstants.SKU_PRICE_PREFIX + itemId, "");
    }

    @Override
    public void refreshItems() {
        ensureInitialized();

        try {
            Inventory inventory = iabHelper.queryInventory(true, Arrays.asList(BillingConstants.ALL_SKUS));
            if (inventory == null) return;

            for (String sku : BillingConstants.ALL_SKUS) {
                boolean itemPurchased = inventory.hasPurchase(sku)
                        && inventory.getPurchase(sku).getPurchaseState() == ITEM_PURCHASED;
                SharedPreferences.Editor editor = preferences.edit();
                editor.putBoolean(sku, itemPurchased);
                if (inventory.hasDetails(sku)) {
                    editor.putString(PreferencesConstants.SKU_PRICE_PREFIX + sku, inventory.getSkuDetails(sku).getPrice());
                }
                editor.apply();
            }
        } catch (IabException e) {
            DebugLog.logException(e);
        }
    }

    @Override
    public void refreshItemsAsync(final RefreshItemsListener listener) {
        if (singleThreadExecutor == null) {
            throw new IllegalStateException("Single thread executor was not initialized!");
        }
        if (singleThreadExecutor.isShutdown()) {
            return;
        }
        singleThreadExecutor.submit(new Runnable() {
            @Override
            public void run() {
                refreshItems();
                notifyItemsRefreshFinished(listener);
            }
        });
    }

    @Override
    public void startBillingWorkflowForItem(Activity activity, final String itemId, final IabHelper.OnIabPurchaseFinishedListener listener) {
        ensureInitialized();

        IabHelper.OnIabPurchaseFinishedListener purchaseFinishedListener = new IabHelper.OnIabPurchaseFinishedListener() {
            @Override
            public void onIabPurchaseFinished(final IabResult result, final Purchase purchase) {
                refreshItems();
                if (result.getResponse() == IabHelper.BILLING_RESPONSE_RESULT_ITEM_ALREADY_OWNED
                        || result.getResponse() == IabHelper.BILLING_RESPONSE_RESULT_OK) {
                    preferences.edit()
                            .putBoolean(itemId, true)
                            .apply();
                }
                notifyItemPurchaseFinished(result, purchase, listener);
            }
        };

        switch (itemId) {
            case BillingConstants.PRO_PACK_SUBSCRIPTION_SKU:
                iabHelper.launchSubscriptionPurchaseFlow(activity, itemId, PURCHASE_REQUEST_CODE, purchaseFinishedListener);
                break;
            default:
                iabHelper.launchPurchaseFlow(activity, itemId, PURCHASE_REQUEST_CODE, purchaseFinishedListener);
        }
    }

    @Override
    public boolean handleResult(int requestCode, int resultCode, Intent data) {
        if (!isInitialized()) {
            return false;
        }

        return iabHelper.handleActivityResult(requestCode, resultCode, data);
    }

    @Override
    public void dispose() {
        singleThreadExecutor.submit(new Runnable() {
            @Override
            public void run() {
                if (iabHelper != null) {
                    iabHelper.dispose();
                    iabHelper = null;
                }
            }
        });
        singleThreadExecutor.shutdown();
    }

    private void notifySetupFinished(final IabHelper.OnIabSetupFinishedListener listener, final IabResult result) {
        if (listener != null) {
            MAIN_LOOPER_HANDLER.post(new Runnable() {
                @Override
                public void run() {
                    listener.onIabSetupFinished(result);
                }
            });
        }
    }

    private void notifyItemsRefreshFinished(final RefreshItemsListener listener) {
        if (listener != null) {
            MAIN_LOOPER_HANDLER.post(new Runnable() {
                @Override
                public void run() {
                    listener.onItemsRefreshFinished();
                }
            });
        }
    }

    private void notifyItemPurchaseFinished(final IabResult result, final Purchase purchase, final IabHelper.OnIabPurchaseFinishedListener listener) {
        if (listener != null) {
            MAIN_LOOPER_HANDLER.post(new Runnable() {
                @Override
                public void run() {
                    listener.onIabPurchaseFinished(result, purchase);
                }
            });
        }
    }

    private void ensureInitialized() {
        if (!isInitialized()) {
            throw new IllegalStateException("IabHelper was not initialized!");
        }
    }

    /**
     * Notified when items are refreshed
     */
    public interface RefreshItemsListener {

        public void onItemsRefreshFinished();

    }
}
