package net.doo.snap.lib.billing;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Handler;
import android.os.Looper;

import com.amazon.device.iap.PurchasingListener;
import com.amazon.device.iap.PurchasingService;
import com.amazon.device.iap.model.Product;
import com.amazon.device.iap.model.ProductDataResponse;
import com.amazon.device.iap.model.PurchaseResponse;
import com.amazon.device.iap.model.PurchaseUpdatesResponse;
import com.amazon.device.iap.model.Receipt;
import com.amazon.device.iap.model.UserDataResponse;
import com.google.inject.Inject;

import net.doo.snap.lib.PreferencesConstants;
import net.doo.snap.lib.util.billing.IabHelper;
import net.doo.snap.lib.util.billing.IabResult;

import java.util.HashSet;

/**
 * Amazon In-app purchase implementation
 */
public class AmazonBillingManagerImpl implements BillingManager {
    private static final Handler MAIN_LOOPER_HANDLER = new Handler(Looper.getMainLooper());

    private boolean initialized = false;

    @Inject
    private SharedPreferences preferences;

    @Inject
    public AmazonBillingManagerImpl() {}

    private RefreshItemsListener refreshItemsListener;
    private IabHelper.OnIabPurchaseFinishedListener purchaseFinishedListener;
    private IabHelper.OnIabSetupFinishedListener setupFinishedListener;


    private PurchasingListener purchasingListener = new PurchasingListener() {
        @Override
        public void onUserDataResponse(UserDataResponse userDataResponse) {}

        @Override
        public void onProductDataResponse(ProductDataResponse productDataResponse) {
            SharedPreferences.Editor editor = preferences.edit();
            for (String sku : productDataResponse.getProductData().keySet()) {
                Product product = productDataResponse.getProductData().get(sku);
                editor.putString(PreferencesConstants.SKU_PRICE_PREFIX + sku, product.getPrice());
            }
            editor.apply();
            refreshItems();
        }

        @Override
        public void onPurchaseResponse(PurchaseResponse purchaseResponse) {
            IabResult result = null;
            Receipt receipt = purchaseResponse.getReceipt();

            switch (purchaseResponse.getRequestStatus()) {
                case SUCCESSFUL:
                    result = new IabResult(IabHelper.BILLING_RESPONSE_RESULT_OK, null);
                    if (receipt != null) {
                        SharedPreferences.Editor editor = preferences.edit();
                        editor.putBoolean(receipt.getSku(), !receipt.isCanceled());
                        editor.apply();
                    }
                    break;
                case ALREADY_PURCHASED:
                    result = new IabResult(IabHelper.BILLING_RESPONSE_RESULT_ITEM_ALREADY_OWNED, null);
                    if (receipt != null) {
                        SharedPreferences.Editor editor = preferences.edit();
                        editor.putBoolean(receipt.getSku(), !receipt.isCanceled());
                        editor.apply();
                    }
                    break;
                case FAILED:
                    result = new IabResult(IabHelper.BILLING_RESPONSE_RESULT_USER_CANCELED, null);
                    break;
                case INVALID_SKU:
                    result = new IabResult(IabHelper.BILLING_RESPONSE_RESULT_ITEM_UNAVAILABLE, null);
                    break;
                case NOT_SUPPORTED:
                    result = new IabResult(IabHelper.BILLING_RESPONSE_RESULT_BILLING_UNAVAILABLE, null);
                    break;
            }

            if (purchaseFinishedListener != null) {
                purchaseFinishedListener.onIabPurchaseFinished(result, null);
            }

            purchaseFinishedListener = null;
        }

        @Override
        public void onPurchaseUpdatesResponse(PurchaseUpdatesResponse purchaseUpdatesResponse) {
            for (String sku : BillingConstants.ALL_SKUS) {
                SharedPreferences.Editor editor = preferences.edit();
                editor.putBoolean(sku, false);
                editor.apply();
            }

            for (Receipt receipt : purchaseUpdatesResponse.getReceipts()) {
                SharedPreferences.Editor editor = preferences.edit();
                editor.putBoolean(receipt.getSku(), !receipt.isCanceled());
                editor.apply();
            }

            if (refreshItemsListener != null) {
                refreshItemsListener.onItemsRefreshFinished();
                refreshItemsListener = null;
            }

            notifySetupFinished();
        }
    };

    @Override
    public void initialize(Context context, final IabHelper.OnIabSetupFinishedListener listener) {
        PurchasingService.registerListener(context, purchasingListener);

        setupFinishedListener = listener;
        HashSet<String> skus = new HashSet<>();

        for (String sku : BillingConstants.ALL_SKUS) {
            skus.add(sku);
        }

        PurchasingService.getProductData(skus);
    }

    @Override
    public boolean isInitialized() {
        return initialized;
    }

    @Override
    public boolean isItemAvailable(String itemId) {
        return preferences.getBoolean(itemId, false);
    }

    @Override
    public boolean isBillingContentAvailable(BillingContent billingContent) {
        return BillingContent.isConentAvailable(billingContent, this);
    }

    @Override
    public String getItemPrice(String itemId) {
        return preferences.getString(PreferencesConstants.SKU_PRICE_PREFIX + itemId, "");
    }

    @Override
    public void refreshItems() {
        PurchasingService.getPurchaseUpdates(true);
    }

    @Override
    public void refreshItemsAsync(RefreshItemsListener listener) {
        refreshItemsListener = listener;
        refreshItems();
    }

    @Override
    public void startBillingWorkflowForItem(Activity activity, String itemId, IabHelper.OnIabPurchaseFinishedListener  listener) {
        purchaseFinishedListener = listener;
        PurchasingService.purchase(itemId);
    }

    @Override
    public boolean handleResult(int requestCode, int resultCode, Intent data) {
        return false;
    }

    @Override
    public void dispose() {
    }

    private void notifySetupFinished() {
        initialized = true;
        if (setupFinishedListener != null) {
            MAIN_LOOPER_HANDLER.post(new Runnable() {
                @Override
                public void run() {
                    setupFinishedListener.onIabSetupFinished(null);
                }
            });
        }
    }
}
