package net.doo.snap.lib.edit;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.BitmapShader;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.Shader;
import android.graphics.drawable.BitmapDrawable;
import android.util.AttributeSet;
import android.view.View;
import android.widget.ImageView;

import net.doo.snap.lib.R;
import net.doo.snap.lib.util.ThemesHelper;

/**
 * View that draws magnifier. Magnifier should be properly set up before draw
 *
 * @see net.doo.snap.lib.edit.DrawMagnifierListener
 */
public class MagnifierView extends View implements DrawMagnifierListener {
    private Magnifier magnifier;
    private PointF zoomPoint;
    private Bitmap magnifierBitmap;

    public MagnifierView(Context context, AttributeSet attrs) {
        super(context, attrs);
        magnifier = new Magnifier();
    }

    @Override
    public void drawMagnifier(PointF zoomPoint) {
        this.zoomPoint = zoomPoint;
        invalidate();
    }

    @Override
    public void eraseMagnifier() {
        zoomPoint = null;
        invalidate();
    }

    /**
     * Setup magnifier using given {@link android.widget.ImageView}
     * @param view
     */
    public void setupMagnifier(ImageView view) {
        magnifier.setup(view);
    }

    @Override
    protected void dispatchDraw(Canvas canvas) {
        super.dispatchDraw(canvas);
        if (zoomPoint != null) {
            magnifier.draw(canvas, zoomPoint);
        }
    }

    @Override
    public void onAttachedToWindow() {
        super.onAttachedToWindow();
        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inPreferredConfig = Bitmap.Config.ARGB_8888;
        magnifierBitmap = BitmapFactory.decodeResource(getResources(), ThemesHelper.getResourceId(getContext(), R.attr.ui_crop_magnifier), options);
    }

    @Override
    public void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        magnifierBitmap.recycle();
    }

    @Override
    protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
        super.onLayout(changed, left, top, right, bottom);
        magnifier.setViewDimensions(getWidth(), getHeight());
    }

    /**
     * Helper to draw zoomed part of image in touched area
     */
    private class Magnifier {
        private final float ZOOM = 2f;

        private float bitmapScale;

        private int viewWidth;
        private int viewHeight;
        private BitmapShader shader;
        private Paint shaderPaint;
        private Matrix matrix;

        private Paint magnifierPaint;
        private Paint magnifierCrossPaint;

        private float radius;
        private float crossSize;
        private float margin;

        private float offsetX;
        private float offsetY;

        private float rotation;

        private BitmapDrawable drawable;


        /**
         * After calling a constructor you need to call {@link #setup}
         * when bitmap is set and size of View is calculated
         */
        Magnifier() {
            magnifierPaint = new Paint();
            magnifierPaint.setColor(getResources().getColor(R.color.edit_polygon_magnifier));
            magnifierPaint.setStyle(Paint.Style.STROKE);
            magnifierPaint.setStrokeWidth(getResources().getDimension(R.dimen.magnifier_stroke_width));
            magnifierPaint.setAntiAlias(true);

            magnifierCrossPaint = new Paint();
            magnifierCrossPaint.setColor(getResources().getColor(R.color.edit_polygon_magnifier));
            magnifierCrossPaint.setStyle(Paint.Style.STROKE);
            magnifierCrossPaint.setStrokeWidth(getResources().getDimension(R.dimen.magnifier_cross_stroke_width));
            magnifierCrossPaint.setAntiAlias(true);

            shaderPaint = new Paint();

            matrix = new Matrix();

            radius = getResources().getDimension(R.dimen.magnifier_raduis);
            crossSize = getResources().getDimension(R.dimen.magnifier_cross_size);
            margin = getResources().getDimension(R.dimen.magnifier_margin);
        }

        void setViewDimensions(int width, int height) {
            viewWidth = width;
            viewHeight = height;

            if (drawable != null && width != 0 && height != 0) {
                bitmapScale = Math.max((float) drawable.getIntrinsicWidth() / viewWidth, (float) drawable.getIntrinsicHeight() / viewHeight);
                offsetX = (viewWidth - drawable.getIntrinsicWidth() / bitmapScale) / 2;
                offsetY = (viewHeight - drawable.getIntrinsicHeight() / bitmapScale) / 2;
            }
        }

        /**
         * Creates {@link android.graphics.BitmapShader} which will be drawn on screen
         *
         * @param view Image view with {@link android.graphics.drawable.BitmapDrawable} set in it
         */
        void setup(ImageView view) {
            if (shader == null && view.getDrawable() != null) {
                drawable = (BitmapDrawable) view.getDrawable();
                shader = new BitmapShader(drawable.getBitmap(), Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);
                shaderPaint.setShader(shader);
                rotation = view.getRotation();

                setViewDimensions(getWidth(), getHeight());
            }
        }

        /**
         * Draws magnifier on canvas for given zoom point
         *
         * @param canvas    canvas to draw magnifier
         * @param zoomPoint coordinates of touched point on view subtracted by {@link #offsetX} and {@link #offsetY}
         */
        void draw(Canvas canvas, PointF zoomPoint) {
            // left top position of magnifier
            float startX = margin;
            float startY = margin;

            // if touch event is in top left corner them move magnifier to top right
            if (isInLeftTop(zoomPoint)) {
                startX = getWidth() - radius * 2 - margin;
            }

            // set zoom ratio relatively to original bitmap resolution
            matrix.setScale(ZOOM, ZOOM);
            // translate canvas on original bitmap to match center of magnifier and zoomPoint coordinates
            // radius - magnifier radius
            // bitmapScale - how much original bitmap is larger than scaled image in ImageView
            // zoompoint - coordinates from 0 to scaled image dimensions.
            matrix.postTranslate(-zoomPoint.x * bitmapScale * ZOOM + radius + startX, -zoomPoint.y * bitmapScale * ZOOM + radius + startY);
            shader.setLocalMatrix(matrix);

            canvas.save();
            canvas.rotate(rotation, startX + radius, startY + radius);
            // Draw magnifier using shaderPaint as Paint
            canvas.drawCircle(startX + radius, startY + radius, radius, shaderPaint);
            canvas.restore();

            // Draw frame around magnifier
            canvas.drawBitmap(magnifierBitmap, startX, startY, null);
        }

        private boolean isInLeftTop(PointF point) {
            int quarter = 1;
            if (point.x + offsetX < viewWidth >> 1) {
                if (point.y + offsetY < viewHeight >> 1) {
                    quarter = 1;
                } else {
                    quarter = 4;
                }
            } else {
                if (point.y + offsetY < viewHeight >> 1) {
                    quarter = 2;
                } else {
                    quarter = 3;
                }
            }

            // compensate rotation
            quarter -= (360 - rotation) % 360 / 90;

            return quarter == 1;
        }
    }

}
