package net.doo.snap.lib.ui.tutorial;

import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.support.v4.app.FragmentActivity;
import android.support.v4.app.FragmentTransaction;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;

import com.google.inject.Inject;

import net.doo.snap.lib.R;
import net.doo.snap.lib.util.ThemesHelper;
import net.doo.snap.lib.util.ui.ViewUtils;

import org.jetbrains.annotations.NotNull;

import roboguice.fragment.RoboDialogFragment;

/**
 * Base class for fragments which should be shown only once per application installation.
 * Controls whether it will be displayed or not, depending on {@link android.content.SharedPreferences}
 */
public abstract class OneTimeFragment extends RoboDialogFragment {

    private static final int FADE_IN_DURATION = 600;

    @Inject
    private SharedPreferences sharedPreferences;
    @Inject
    private Context context;

    @Override
    public Dialog onCreateDialog(Bundle savedInstanceState) {
        Dialog dialog = super.onCreateDialog(savedInstanceState);

        Window window = dialog.getWindow();
        window.setBackgroundDrawableResource(ThemesHelper.getResourceId(context, R.attr.main_color_transparent));
        window.clearFlags(WindowManager.LayoutParams.FLAG_DIM_BEHIND);

        return dialog;
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        setStyle(STYLE_NO_FRAME, android.R.style.Theme_DeviceDefault_NoActionBar);
        super.onCreate(savedInstanceState);
    }

    @Override
    public void onViewCreated(View view, Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        final View cancelView = getCancelView();

        if (cancelView != null) {
            getCancelView().setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    consume();
                    dismissAllowingStateLoss();
                }
            });
        }

        View decorView = getDialog().getWindow().getDecorView();
        decorView.setAlpha(ViewUtils.ALPHA_TRANSPARENT);
        decorView.animate()
                .alpha(ViewUtils.ALPHA_DEFAULT)
                .setDuration(FADE_IN_DURATION);
    }

    /**
     * Cancels current tutorial so it won't show up again
     */
    protected void consume() {
        sharedPreferences.edit()
                .putBoolean(getDialogId(), true)
                .commit();
    }

    @Override
    public void onCancel(DialogInterface dialog) {
        super.onCancel(dialog);
        consume();
    }

    /**
     * @return {@code true} if dialog should be shown, {@code false} otherwise
     */
    protected boolean shouldBeShown(String tutorialId, FragmentActivity activity, SharedPreferences sharedPreferences) {
        return !sharedPreferences.getBoolean(tutorialId, false)
                && activity.getSupportFragmentManager().findFragmentByTag(tutorialId) == null;

    }

    /**
     * Displays dialog if it wasn't shown to user before.
     */
    public final void showOnce(@NotNull FragmentActivity activity, @NotNull SharedPreferences sharedPreferences) {
        final String tutorialId = getDialogId();

        if (!shouldBeShown(tutorialId, activity, sharedPreferences)) {
            return;
        }

        FragmentTransaction ft = activity.getSupportFragmentManager().beginTransaction();
        ft.add(this, tutorialId);
        ft.commitAllowingStateLoss();
    }

    /**
     * @return {@link String} identifier of this dialog, which will be used as key for {@link android.content.SharedPreferences}
     */
    protected abstract String getDialogId();

    /**
     * @return {@link android.view.View} which will dismiss this tutorial on click
     */
    protected View getCancelView() {
        return null;
    }

}
