package net.doo.snap.lib.persistence;

import android.os.Parcel;
import android.os.Parcelable;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Contains {@link Page}s and naming metadata necessary for document creation
 *
 * @see net.doo.snap.lib.persistence.PageFactory
 */
public class DocumentDraft implements Parcelable {
    private final ArrayList<Page> pages;
    private String documentName;

    public DocumentDraft(Page... pages) {
        this.pages = new ArrayList<Page>();
        Collections.addAll(this.pages, pages);
    }

    /**
     * Adds page in the end
     *
     * @param page
     */
    public void addPage(Page page) {
        pages.add(page);
    }

    /**
     * Adds page to given position
     *
     * @param position
     * @param page
     */
    public void addPage(int position, Page page) {
        pages.add(position, page);
    }

    /**
     * Replaces page at given position with new page
     *
     * @param position
     * @param page
     * @throws NoPageByIndexException
     */
    public void replacePage(int position, Page page) throws NoPageByIndexException {
        if (pages.size() == 0 || position < 0 || position >= pages.size()) {
            throw new NoPageByIndexException();
        }
        pages.remove(position);
        if (position == pages.size()) {
            pages.add(page);
        } else {
            pages.add(position, page);
        }
    }

    /**
     * @param position
     * @return {@link net.doo.snap.lib.persistence.Page} at given position
     * @throws NoPageByIndexException
     */
    public Page getPage(int position) throws NoPageByIndexException {
        if (pages.size() == 0 || position < 0 || position >= pages.size()) {
            throw new NoPageByIndexException();
        }
        return pages.get(position);
    }

    /**
     * Deletes {@link net.doo.snap.lib.persistence.Page} at given position
     *
     * @param position
     * @throws NoPageByIndexException
     */
    public void deletePage(int position) throws NoPageByIndexException {
        if (pages.size() == 0 || position < 0 || position >= pages.size()) {
            throw new NoPageByIndexException();
        }
        pages.remove(position);
    }

    /**
     * Deletes {@link net.doo.snap.lib.persistence.Page} if possible
     *
     * @param page {@link net.doo.snap.lib.persistence.Page} to delete
     */
    public void deletePage(Page page) {
        pages.remove(page);
    }

    /**
     * @param page
     * @return position of given page
     */
    public int getPosition(Page page) {
        return pages.indexOf(page);
    }

    /**
     * @return count of pages stored in DocumentDraft
     */
    public int size() {
        return pages.size();
    }

    /**
     * @return {@code true} if the holder has more than 0 pages, {@code false} otherwise
     */
    public boolean isEmpty() {
        return pages.isEmpty();
    }

    /**
     * @return unmodifiable list of {@link net.doo.snap.lib.persistence.Page}s snapped in the current session
     */
    public List<Page> getPages() {
        return Collections.unmodifiableList(pages);
    }

    /**
     * @return Document name to be saved on file system
     */
    public String getDocumentName() {
        return documentName;
    }

    /**
     * @param documentName Document name to be saved on file system
     */
    public void setDocumentName(String documentName) {
        this.documentName = documentName;
    }

    protected DocumentDraft(Parcel in) {
        if (in.readByte() == 0x01) {
            pages = new ArrayList<Page>();
            in.readList(pages, Page.class.getClassLoader());
        } else {
            pages = null;
        }
        documentName = in.readString();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        if (pages == null) {
            dest.writeByte((byte) (0x00));
        } else {
            dest.writeByte((byte) (0x01));
            dest.writeList(pages);
        }
        dest.writeString(documentName);
    }

    @SuppressWarnings("unused")
    public static final Parcelable.Creator<DocumentDraft> CREATOR = new Parcelable.Creator<DocumentDraft>() {
        @Override
        public DocumentDraft createFromParcel(Parcel in) {
            return new DocumentDraft(in);
        }

        @Override
        public DocumentDraft[] newArray(int size) {
            return new DocumentDraft[size];
        }
    };


    /**
     * Thrown if page by index does not exist
     */
    public static class NoPageByIndexException extends IndexOutOfBoundsException {
    }
}
