package net.doo.snap.lib.util.ui;

import android.animation.Animator;
import android.app.Activity;
import android.content.Context;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.TextView;

import net.doo.snap.lib.R;

/**
 * Shows toast message with UNDO button
 */
public class UndoToast {

    /**
     * Listener which will be invoked when Undo button will be pressed
     */
    public interface OnUndoListener {

        /**
         * Invoked when Undo button is pressed
         */
        void onUndo();

    }

    public static final int DURATION_INFINITE = -1;
    public static final int DURATION_NORMAL_MS = 3000;

    private final Activity activity;
    private final ToastView toastView;
    private boolean animate = true;
    private int duration = DURATION_INFINITE;
    private OnUndoListener listener;

    private boolean shown = false;
    private boolean dismissed = false;

    private final Runnable dismissRunnable = new Runnable() {
        @Override
        public void run() {
            if (toastView.getParent() != null) {
                dismissToast();
            }
        }
    };

    public UndoToast(Activity activity) {
        this.activity = activity;

        toastView = new ToastView(activity);
    }

    /**
     * Detaches toast from {@link android.app.Activity}
     *
     * @param activity from which toast will be detached
     * @return {@code true} if toast was detached. {@code false} if toast wasn't found in {@link android.app.Activity}
     * @throws java.lang.NullPointerException if {@link android.app.Activity} is {@code null}
     */
    public static boolean detachToast(Activity activity) throws NullPointerException {
        ToastView previous = (ToastView) activity.findViewById(R.id.toast);
        if (previous != null && previous.getParent() instanceof ViewGroup) {
            ((ViewGroup) previous.getParent()).removeView(previous);
            return true;
        }

        return false;
    }

    /**
     * @param resId message to be displayed
     * @return same {@link UndoToast}
     */
    public UndoToast message(int resId) {
        ensureNotShown();

        toastView.message.setText(resId);
        return this;
    }

    /**
     * @param message to be displayed
     * @return same {@link UndoToast}
     */
    public UndoToast message(String message) {
        ensureNotShown();

        toastView.message.setText(message);
        return this;
    }

    /**
     * @param listener {@link UndoToast.OnUndoListener} which will be invoked
     *                 on Undo action
     * @return same {@link UndoToast}
     */
    public UndoToast listener(OnUndoListener listener) {
        ensureNotShown();

        this.listener = listener;
        return this;
    }

    /**
     * @param animate {@code true} to use animation for showing/hiding of toast. {@code false} otherwise.
     * @return same {@link UndoToast}
     */
    public UndoToast animate(boolean animate) {
        ensureNotShown();

        this.animate = animate;
        return this;
    }

    /**
     * @param duration for which toast will remain visible
     * @return same {@link UndoToast}
     */
    public UndoToast duration(int duration) {
        ensureNotShown();

        if (duration < 0) {
            duration = DURATION_INFINITE;
        }

        this.duration = duration;
        return this;
    }

    /**
     * Finalizes toast creation and shows view
     */
    public void show() {
        ensureNotShown();
        boolean detached = detachToast();
        setupListener();

        attachToast();

        if (animate && !detached) {
            animateToastIn();
        }

        shown = true;
    }

    private void setupListener() {
        if (listener != null) {
            toastView.undoButton.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    if (dismissed) {
                        return;
                    }

                    listener.onUndo();
                    dismissToast();
                }
            });
        }
    }

    private void attachToast() {
        ((ViewGroup) activity.findViewById(android.R.id.content)).addView(toastView);

        if (duration != DURATION_INFINITE) {
            scheduleDetachment();
        }
    }

    private boolean detachToast() {
        return detachToast(activity);
    }

    private void scheduleDetachment() {
        toastView.postDelayed(dismissRunnable, duration);
    }

    private void dismissToast() {
        dismissed = true;
        if (animate) {
            animateToastOut();
        } else {
            detachToast();
        }
    }

    private void animateToastIn() {
        toastView.setAlpha(ViewUtils.ALPHA_TRANSPARENT);
        toastView.animate().alpha(ViewUtils.ALPHA_DEFAULT);
    }

    private void animateToastOut() {
        toastView
                .animate()
                .alpha(ViewUtils.ALPHA_TRANSPARENT)
                .setListener(new ViewUtils.DefaultAnimationListener() {
                    @Override
                    public void onAnimationFinished(Animator animator) {
                        detachToast();
                    }
                });
    }

    private void ensureNotShown() {
        if (shown) {
            throw new IllegalStateException("Toast was already shown");
        }
    }

    /**
     * Undo toast itself
     */
    public static class ToastView extends FrameLayout {

        public final TextView message;
        public final View undoButton;

        public ToastView(Context context) {
            super(context);
            LayoutInflater.from(context).inflate(R.layout.undo_toast, this);

            message = (TextView) findViewById(R.id.message);
            undoButton = findViewById(R.id.undo_button);

            setId(R.id.toast);
        }

    }

}
