package net.doo.snap.lib.snap.camera;

import android.content.Context;
import android.content.SharedPreferences;
import android.graphics.ImageFormat;
import android.graphics.PointF;
import android.hardware.Camera;
import android.os.Handler;
import android.os.Looper;

import com.commonsware.cwac.camera.CameraUtils;
import com.commonsware.cwac.camera.PictureTransaction;
import com.commonsware.cwac.camera.SimpleCameraHost;
import com.google.inject.Inject;

import net.doo.snap.lib.PreferencesConstants;
import net.doo.snap.lib.detector.CameraDetectorListener;
import net.doo.snap.lib.detector.ContourDetector;
import net.doo.snap.lib.detector.DetectionResult;
import net.doo.snap.lib.snap.camera.barcode.BarcodeDetector;
import net.doo.snap.lib.util.log.DebugLog;
import net.doo.snap.lib.util.snap.Utils;

import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * Implementation of {@link com.commonsware.cwac.camera.CameraHost}
 * to handle callbacks from {@link com.commonsware.cwac.camera.CameraView}
 */
public class SnapCameraHost extends SimpleCameraHost implements Camera.PreviewCallback {
    private static final int MAX_FOCUS_FAIL_COUNT = 3;

    @Inject
    private BarcodeDetector barcodeDetector;
    @Inject
    private SharedPreferences preferences;

    private CameraPreviewFragment cameraPreviewFragment;
    private final Executor executor;
    private final ContourDetector detector;
    private boolean useContourDetection = true;
    private CameraDetectorListener cameraDetectorListener;

    private int orientation;
    private int width;
    private int height;

    private int previewWidth;
    private int previewHeight;

    private int failedFocusCount = 0;

    private Handler mainLooperHandler = new Handler(Looper.getMainLooper());

    @Inject
    public SnapCameraHost(Context context) {
        super(context);
        this.detector = new ContourDetector();
        executor = new ThreadPoolExecutor(
                1, 1, 0L,
                TimeUnit.MILLISECONDS,
                new LinkedBlockingDeque<Runnable>(1),
                new ThreadPoolExecutor.DiscardOldestPolicy()
        );
    }

    /**
     * Attaches host to {@link CameraPreviewFragment}
     */
    public void setCameraPreviewFragment(final CameraPreviewFragment cameraPreviewFragment) {
        this.cameraPreviewFragment = cameraPreviewFragment;
        this.cameraDetectorListener = new CameraDetectorListener() {
            @Override
            public void onDetectionOK(List<PointF> polygon) {
                if (cameraPreviewFragment != null && cameraPreviewFragment.getDetectorListener() != null) {
                    cameraPreviewFragment.getDetectorListener().onDetectionOK(polygon);
                    cameraPreviewFragment.onPolygonDetected();
                }
            }

            @Override
            public void onBarcodeDetectionOK(String content) {
                if (cameraPreviewFragment != null
                        && cameraPreviewFragment.getActivity() != null) {
                    cameraPreviewFragment.onBarcodeDetected();
                }
            }

            @Override
            public void onDetectionWithError(DetectionResult result, List<PointF> polygon) {
                if (cameraPreviewFragment != null && cameraPreviewFragment.getDetectorListener() != null) {
                    cameraPreviewFragment.getDetectorListener().onDetectionWithError(result, polygon);
                    cameraPreviewFragment.onPolygonDetected();
                }
            }

            @Override
            public void onDetectionFailed(DetectionResult result) {
                if (cameraPreviewFragment != null && cameraPreviewFragment.getDetectorListener() != null) {
                    cameraPreviewFragment.getDetectorListener().onDetectionFailed(result);
                }
            }
        };
    }

    @Override
    public Camera.Parameters adjustPreviewParameters(Camera.Parameters parameters) {
        final boolean flashSupported = Utils.isFlashSupported(parameters);

        mainLooperHandler.post(new Runnable() {
            @Override
            public void run() {
                cameraPreviewFragment.setFlashButtonVisible(flashSupported);
            }
        });

        if (flashSupported) {
            // In case of rotation or fragment resume
            parameters.setFlashMode(cameraPreviewFragment.isFlashEnabled() ? Camera.Parameters.FLASH_MODE_TORCH : Camera.Parameters.FLASH_MODE_OFF);
        }
        return parameters;
    }

    @Override
    public void onAutoFocus(boolean success, Camera camera) {
        if (cameraPreviewFragment.isRefocusing()) {
            cameraPreviewFragment.finishRefocusing();
            return;
        }

        boolean isAutoSnapEnabled = cameraPreviewFragment.isAutosnapEnabled();
        // Snap only if contour detection is enabled, so preview is visible and active
        if (useContourDetection || !isAutoSnapEnabled) {
            // If autoSnap enabled and focus is not succes we increase failed focus counter
            if (!success && isAutoSnapEnabled && failedFocusCount < MAX_FOCUS_FAIL_COUNT) {
                failedFocusCount++;
                camera.cancelAutoFocus();
            } else {
                failedFocusCount = 0;
            }
        }
    }

    public void takePicture() {
        useContourDetection = false;

        cameraPreviewFragment.onStartPictureProcessing();
        try {
            cameraPreviewFragment.takePicture(
                    new PictureTransaction(this)
                            .needBitmap(false)
                            .needByteArray(true)
                            .useSingleShotMode(true)
            );
        } catch (RuntimeException e) {
            DebugLog.logException(e);
        }
    }

    @Override
    public void saveImage(PictureTransaction xact, byte[] image, int imageOrientation) {
        cameraPreviewFragment.onPictureTaken(image, imageOrientation);
        cameraPreviewFragment.onFinishPictureProcessing();
    }

    @Override
    public void onPreviewFrame(final byte[] image, Camera camera) {
        if (useContourDetection) {
            //TODO: update condition !(cameraPreviewFragment.getActivity() instanceof CameraActivity) when lib will be integrated into the app
            barcodeDetector.enableBarcodeScan(cameraPreviewFragment.getActivity() != null
                    && !(cameraPreviewFragment.getActivity() instanceof CameraActivity)
                    && preferences.getBoolean(PreferencesConstants.SCAN_BARCODES, true));

            executor.execute(new DetectionTask(barcodeDetector, detector, image, previewWidth, previewHeight, cameraDetectorListener));
        }
    }

    /**
     * Indicates whether contour detection should be used or not. Notice, that
     * setting this flag to {@code false} doesn't cancels existing tasks, so results might still come.
     *
     * @param useContourDetection
     */
    public void setUseContourDetection(boolean useContourDetection) {
        this.useContourDetection = useContourDetection;
    }

    /**
     * @return whether contour detection should be used or not.
     */
    public boolean useContourDetection() {
        return useContourDetection;
    }

    @Override
    public RecordingHint getRecordingHint() {
        return RecordingHint.STILL_ONLY;
    }

    @Override
    public final Camera.Size getPreviewSize(int displayOrientation, int width, int height, Camera.Parameters parameters) {
        orientation = displayOrientation;
        this.width = width;
        this.height = height;

        Camera.Size previewSize = findPreviewSize(displayOrientation, width, height, parameters);

        previewWidth = previewSize.width;
        previewHeight = previewSize.height;
        return previewSize;
    }

    /**
     * @return preview size which best matches current device configuration
     */
    protected Camera.Size findPreviewSize(int displayOrientation, int width, int height, Camera.Parameters parameters) {
        return CameraUtils.getBestAspectPreviewSize(
                displayOrientation,
                width,
                height,
                parameters
        );
    }

    /**
     * @return newly created buffer for preview callback.
     */
    public byte[] getNewPreviewBuffer() {
        return new byte[previewWidth * previewHeight * ImageFormat.getBitsPerPixel(ImageFormat.NV21) / 8];
    }

    @Override
    public Camera.Size getPictureSize(PictureTransaction xact, Camera.Parameters parameters) {
        return Utils.getLargestPictureSize(parameters);
    }

    @Override
    public boolean useFullBleedPreview() {
        return true;
    }

}
