package io.scanbot.tools.utils.kotlin

/** Proper and **correct** way to get current timestamp.
 * **NOTE:** [System.currentTimeMillis] is **not** recommended for measuring time intervals.
 *
 * See:
 * - [Reddit link](https://www.reddit.com/r/learnjava/comments/1am6c32/why_is_stopwatch_preferred_to/)
 * - [Android Developers](https://developer.android.com/reference/java/lang/System#currentTimeMillis())
 */
fun currentSyncTimeMillis(): Long {
   return (System.nanoTime().toFloat() / 1_000_000).toLong()
}

/** Sugar around [currentSyncTimeMillis] to measure time intervals of [block]'s execution. */
inline fun measureMillisTime(block: () -> Unit): Long {
   val start = currentSyncTimeMillis()
   block()
   return currentSyncTimeMillis() - start
}

fun millisToHumanReadable(millis: Long): String {
   if (millis == 0L) return "0s"

   val millisLeft = millis % 1000
   val secondsTotal = millis / 1000
   val minutesTotal = secondsTotal / 60
   val hours = minutesTotal / 60
   val minutesOfHour = minutesTotal % 60

   val strParts = mutableListOf<String>()

   if (hours != 0L) strParts.add("${hours}h")
   if (minutesOfHour != 0L || hours != 0L) strParts.add("${minutesOfHour}m")
   if (secondsTotal % 60 != 0L || minutesOfHour != 0L || hours != 0L) strParts.add("${secondsTotal % 60}s")
   if (millisLeft != 0L) strParts.add("${millisLeft}ms")

   return strParts.joinToString(" ")
}
