package io.scanbot.shoeboxed;

import android.app.Activity;
import android.content.Intent;
import android.os.Bundle;

/**
 * Helper class for dealing with Shoeboxed authorization.
 * <p/>
 * Usage is following:
 * 1. Call {@link #launchAuthorization(ApiApplication, android.app.Activity, int)} from your {@link android.app.Activity}
 * 2. After user will complete authorization, you'll get result in {@link android.app.Activity#onActivityResult(int, int, android.content.Intent)}
 * 3. Make sure that request code that you specified in {@link #launchAuthorization(ApiApplication, android.app.Activity, int)} matches
 * request code in {@link android.app.Activity#onActivityResult(int, int, android.content.Intent)}.
 * 4. If it matches, redirect call to {@link #handleAuthorizationResult(int, android.content.Intent)}
 * 5. You'll receive tokens if authorization was successful or {@code null} if it failed
 * <p/>
 * Class does not hold any state, so it can be safely recreated each time it is needed.
 */
public class ShoeboxedAuthHelper {

    /**
     * Launches authorization process
     *
     * @param apiApplication Shoeboxed API client
     * @param context        {@link android.app.Activity} from which authorization started
     * @param requestCode    request code for {@link android.app.Activity#startActivityForResult(android.content.Intent, int)}
     */
    public void launchAuthorization(ApiApplication apiApplication, Activity context, int requestCode) {
        Intent intent = new Intent(context, ShoeboxedAuthActivity.class);
        intent.putExtra(ShoeboxedAuthActivity.ARG_API_APPLICATION, apiApplication);

        context.startActivityForResult(intent, requestCode);
    }

    /**
     * Handles result of authorization.
     *
     * @param resultCode result code from {@link android.app.Activity#onActivityResult(int, int, android.content.Intent)}
     * @param intent     {@link android.content.Intent} from {@link android.app.Activity#onActivityResult(int, int, android.content.Intent)}
     * @return {@link io.scanbot.shoeboxed.AuthHandler.Tokens} or {@code null} if authorization failed
     */
    public AuthHandler.Tokens handleAuthorizationResult(int resultCode, Intent intent) {
        if (resultCode != Activity.RESULT_OK) {
            return null;
        }

        final Bundle extras = intent.getExtras();

        if (extras == null || !extras.containsKey(ShoeboxedAuthActivity.RESULT_TOKENS)) {
            return null;
        }

        return (AuthHandler.Tokens) extras.getSerializable(ShoeboxedAuthActivity.RESULT_TOKENS);
    }

}
