package net.doo.snap.lib.persistence;

import android.graphics.PointF;
import android.os.Parcel;
import android.os.Parcelable;

/**
 * Represents a digital signature for the user physical signature
 *
 * @see net.doo.snap.lib.persistence.Page
 */
public class Signature implements Parcelable {
    private String id;
    private PointF origin = new PointF(0, 0);
    private float scale = 1.0f;
    private String imagePath;

    /**
     * @return signature unique id
     */
    public String getId() {
        return id;
    }

    /**
     * sets signature id
     *
     * @param id
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * @return {@link android.graphics.PointF} signature top left corner relative position on the {@link net.doo.snap.lib.persistence.Page},
     * by default (0, 0)
     */
    public PointF getOrigin() {
        return origin == null ? new PointF(0, 0) : origin;
    }

    /**
     * Sets new signature top left corner relative position on the {@link net.doo.snap.lib.persistence.Page}
     *
     * @param origin
     */
    public void setOrigin(PointF origin) {
        this.origin = origin;
    }

    /**
     * @return signature scale on the {@link net.doo.snap.lib.persistence.Page}, by default {@code 1.0}
     */
    public float getScale() {
        return scale;
    }

    /**
     * Sets signature scale on the {@link net.doo.snap.lib.persistence.Page}. If scale is {@code < 1.0} then the signature is downscaled, otherwise increased.
     *
     * @param scale
     * @throws java.lang.IllegalStateException if the scale is {@code <= 0}
     */
    public void setScale(float scale) {
        if (scale <= 0) {
            throw new IllegalStateException("scale should be > 0");
        }
        this.scale = scale;
    }

    /**
     * @return path to the image of this {@link net.doo.snap.lib.persistence.Signature}. Default is {@code null}.
     */
    public String getImagePath() {
        return imagePath;
    }

    /**
     * @param imagePath path to the image of this {@link net.doo.snap.lib.persistence.Signature}. Can be {@code null}.
     */
    public void setImagePath(String imagePath) {
        this.imagePath = imagePath;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof Signature)) return false;
        return id.equals(((Signature) o).id);
    }

    @Override
    public int hashCode() {
        return id.hashCode();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.id);
        dest.writeParcelable(this.origin, 0);
        dest.writeFloat(this.scale);
        dest.writeString(this.imagePath);
    }

    public Signature() {
    }

    private Signature(Parcel in) {
        this.id = in.readString();
        this.origin = in.readParcelable(PointF.class.getClassLoader());
        this.scale = in.readFloat();
        this.imagePath = in.readString();
    }

    public static Parcelable.Creator<Signature> CREATOR = new Parcelable.Creator<Signature>() {
        public Signature createFromParcel(Parcel source) {
            return new Signature(source);
        }

        public Signature[] newArray(int size) {
            return new Signature[size];
        }
    };
}
