package net.doo.snap.lib.snap;

import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentManager;
import android.support.v4.app.FragmentTransaction;
import android.support.v4.view.PagerAdapter;
import android.view.View;
import android.view.ViewGroup;

import com.google.inject.Inject;

import net.doo.snap.lib.R;
import net.doo.snap.lib.persistence.DocumentDraft;
import net.doo.snap.lib.persistence.Page;
import net.doo.snap.lib.persistence.PageFactory;
import net.doo.snap.lib.snap.camera.CameraPreviewFragment;

/**
 * Adapter for the paging content.
 */
public class PreviewFragmentPagerAdapter extends PagerAdapter {

    public static final float FULL_PAGE_WIDTH = 1f;
    public static final float PREVIEW_PAGE_WIDTH = 0.7f;
    private static final String CAMERA_FRAGMENT_TAG = "CAMERA_FRAGMENT_TAG";
    private static final String FRAGMENT_TAG_PREFIX = "doo:";

    private FragmentManager fragmentManager;
    private FragmentTransaction currentTransaction;
    private Fragment currentPrimaryItem;

    private DocumentDraft documentDraft = new DocumentDraft();

    @Inject
    private PageFactory pageFactory;

    @Inject
    public PreviewFragmentPagerAdapter(FragmentManager fragmentManager) {
        this.fragmentManager = fragmentManager;
    }

    /**
     * Return the Fragment associated with a specified position.
     */
    protected Fragment getItem(int position) {
        return position == documentDraft.size()
                ? new CameraPreviewFragment()
                : ImagePreviewFragment.newInstance(documentDraft.getPage(position));
    }

    /**
     * @return {@link net.doo.snap.lib.snap.camera.CameraPreviewFragment} used by adapter
     */
    public CameraPreviewFragment getCameraFragment() {
        return (CameraPreviewFragment) fragmentManager.findFragmentByTag(CAMERA_FRAGMENT_TAG);
    }

    /**
     * @param container {@link android.view.ViewGroup} container of fragment
     * @param position  position from which fragment should be returned
     * @return {@link net.doo.snap.lib.snap.ImagePreviewFragment} at provided position
     * @throws ClassCastException        if fragment at provided position is not {@link net.doo.snap.lib.snap.ImagePreviewFragment} (and, for example, {@link net.doo.snap.lib.snap.camera.CameraPreviewFragment})
     * @throws IndexOutOfBoundsException if position exceeds size of the adapter
     */
    public ImagePreviewFragment getFragment(ViewGroup container, int position) throws ClassCastException, IndexOutOfBoundsException {
        return (ImagePreviewFragment) instantiateItem(container, position);
    }

    /**
     * Return a unique identifier for the item at the given position.
     * <p/>
     * <p>The default implementation returns the given position.
     * Subclasses should override this method if the positions of items can change.</p>
     *
     * @param position Position within this adapter
     * @return Unique identifier for the item at position
     */
    public long getItemId(int position) {
        long id;
        if (position == documentDraft.size()) {
            id = R.id.camera_id;
        } else {
            id = documentDraft.getPage(position).hashCode();
        }

        return id;
    }

    @Override
    public int getItemPosition(Object object) {
    /*
     * Purpose of this method is to check whether an item in the adapter
     * still exists in the dataset and where it should show.
     * For each entry in dataset, request its Fragment.
     *
     * If the Fragment is found, return its (new) position. There's
     * no need to return POSITION_UNCHANGED; ViewPager handles it.
     *
     * If the Fragment passed to this method is not found, remove all
     * references and let the ViewPager remove it from display by
     * by returning POSITION_NONE;
     */

        // Camera is always at the end of ViewPager
        if (object instanceof CameraPreviewFragment) {
            return getCount() - 1;
        }

        ImagePreviewFragment previewFragment = (ImagePreviewFragment) object;

        int pageIndex = documentDraft.getPosition(previewFragment.getPage());
        if (pageIndex != -1) {
            return pageIndex;
        }

        // Let ViewPager remove the Fragment by returning POSITION_NONE.
        return POSITION_NONE;
    }

    /**
     * @param page
     * @return position of provided {@link net.doo.snap.lib.persistence.Page} in adapter
     */
    public int getPagePosition(Page page) {
        return documentDraft.getPosition(page);
    }

    @Override
    public void setPrimaryItem(ViewGroup container, int position, Object object) {
        Fragment fragment = (Fragment) object;
        if (fragment != currentPrimaryItem) {
            if (currentPrimaryItem != null) {
                currentPrimaryItem.setMenuVisibility(false);
                currentPrimaryItem.setUserVisibleHint(false);
            }
            if (fragment != null) {
                fragment.setMenuVisibility(true);
                fragment.setUserVisibleHint(true);
            }

            if (fragment instanceof ImagePreviewFragment) {
                ((ImagePreviewFragment) fragment).setSelected(true);
            }

            if (currentPrimaryItem instanceof ImagePreviewFragment) {
                ((ImagePreviewFragment) currentPrimaryItem).setSelected(false);
            }

            currentPrimaryItem = fragment;
        }
    }

    @Override
    public Fragment instantiateItem(ViewGroup container, int position) {
        verifyTransactionCreated();

        final long itemId = getItemId(position);

        // Do we already have this fragment?
        String name = makeFragmentName(container.getId(), itemId);
        Fragment fragment = fragmentManager.findFragmentByTag(name);
        if (fragment != null) {
            currentTransaction.attach(fragment);
        } else {
            fragment = getItem(position);
            currentTransaction.add(
                    container.getId(),
                    fragment,
                    makeFragmentName(container.getId(), itemId)
            );
        }
        if (fragment != currentPrimaryItem) {
            fragment.setMenuVisibility(false);
            fragment.setUserVisibleHint(false);
        }

        return fragment;
    }

    @Override
    public void destroyItem(ViewGroup container, int position, final Object object) {
        if (object instanceof CameraPreviewFragment) {
            return;
        }

        verifyTransactionCreated();
        currentTransaction.remove((Fragment) object);
    }

    private void verifyTransactionCreated() {
        if (currentTransaction == null) {
            currentTransaction = fragmentManager.beginTransaction();
        }
    }

    @Override
    public void finishUpdate(ViewGroup container) {
        if (currentTransaction != null) {
            currentTransaction.commitAllowingStateLoss();
            currentTransaction = null;
            fragmentManager.executePendingTransactions();
        }
    }

    @Override
    public boolean isViewFromObject(View view, Object object) {
        return ((Fragment) object).getView() == view;
    }

    @Override
    public int getCount() {
        return documentDraft.size() + 1;
    }

    /**
     * @return amount of document pageInfos in adapter. Convenience replacement for {@code getCount() - 1}
     */
    public int getPagesCount() {
        return documentDraft.size();
    }

    @Override
    public float getPageWidth(int position) {
        return position == getCount() - 1
                ? FULL_PAGE_WIDTH
                : PREVIEW_PAGE_WIDTH;
    }

    /**
     * Adds page in the end (before camera preview)
     *
     * @param page
     */
    public void addPage(final Page page) {
        documentDraft.addPage(page);
        notifyDataSetChanged();
    }

    /**
     * Adds new page to adapter
     *
     * @param position position at which page should be inserted
     * @param page     information about page
     */
    public void addPage(int position, Page page) {
        documentDraft.addPage(position, page);
        notifyDataSetChanged();
    }

    /**
     * Deletes page from adapter
     *
     * @param position position of page, which should be deleted
     */
    public void deletePage(int position) {
        documentDraft.deletePage(position);
        notifyDataSetChanged();
    }

    /**
     * Deletes page from adapter
     *
     * @param page {@link net.doo.snap.lib.persistence.Page} to delete
     */
    public void deletePage(Page page) {
        documentDraft.deletePage(page);
        notifyDataSetChanged();
    }

    /**
     * @return container for pages in this adapter
     */
    public DocumentDraft getDocumentDraft() {
        return documentDraft;
    }

    /**
     * Sets container for pages in this adapter
     *
     * @param documentDraft
     */
    public void setDocumentDraft(DocumentDraft documentDraft) {
        this.documentDraft = documentDraft;
    }

    private static String makeFragmentName(int viewId, long id) {
        return id == R.id.camera_id
                ? CAMERA_FRAGMENT_TAG
                : FRAGMENT_TAG_PREFIX + viewId + ":" + id;
    }
}
