package net.doo.snap.lib.ui.util;

import android.annotation.TargetApi;
import android.content.res.Resources;
import android.graphics.Canvas;
import android.graphics.ColorFilter;
import android.graphics.PorterDuff;
import android.graphics.Rect;
import android.graphics.Region;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.util.AttributeSet;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import java.io.IOException;

/**
 * Wraps another {@link android.graphics.drawable.Drawable} and provide a way to perform transformations
 * on it.
 */
public class TransformableDrawable extends Drawable {

    public static final String PROPERTY_ROTATION = "rotation";
    public static final String PROPERTY_SCALE = "scale";

    private Drawable wrappedDrawable;

    private float rotation = 0f;
    private float scale = ViewUtils.SCALE_DEFAULT;

    public TransformableDrawable(Drawable wrappedDrawable) throws NullPointerException {
        if (wrappedDrawable == null) {
            throw new NullPointerException("Wrapped drawable can't be null");
        }

        this.wrappedDrawable = wrappedDrawable;
    }

    /**
     * @return current rotation of the drawable
     */
    public float getRotation() {
        return rotation;
    }

    /**
     * @param rotation of the drawable
     */
    public void setRotation(float rotation) {
        this.rotation = rotation;
        invalidateSelf();
    }

    /**
     * @return current scale of the drawable
     */
    public float getScale() {
        return scale;
    }

    /**
     * @param scale of the drawable
     */
    public void setScale(float scale) {
        this.scale = scale;
        invalidateSelf();
    }

    @Override
    public void draw(Canvas canvas) {
        Rect bounds = getBounds();
        final int halfWidth = bounds.centerX();
        final int halfHeight = bounds.centerY();

        canvas.save();
        canvas.scale(
                scale, scale,
                halfWidth, halfHeight

        );
        canvas.rotate(rotation, halfWidth, halfHeight);

        wrappedDrawable.draw(canvas);

        canvas.restore();
    }

    @Override
    public void setBounds(int left, int top, int right, int bottom) {
        super.setBounds(left, top, right, bottom);
        wrappedDrawable.setBounds(left, top, right, bottom);
    }

    @Override
    public void setBounds(Rect bounds) {
        super.setBounds(bounds);
        wrappedDrawable.setBounds(bounds);
    }

    @Override
    public void setChangingConfigurations(int configs) {
        wrappedDrawable.setChangingConfigurations(configs);
    }

    @Override
    public int getChangingConfigurations() {
        return wrappedDrawable.getChangingConfigurations();
    }

    @Override
    public void setDither(boolean dither) {
        wrappedDrawable.setDither(dither);
    }

    @Override
    public void setFilterBitmap(boolean filter) {
        wrappedDrawable.setFilterBitmap(filter);
    }

    @Override
    public void setAlpha(int alpha) {
        wrappedDrawable.setAlpha(alpha);
    }

    @TargetApi(Build.VERSION_CODES.KITKAT)
    @Override
    public int getAlpha() {
        return wrappedDrawable.getAlpha();
    }

    @Override
    public void setColorFilter(ColorFilter cf) {
        wrappedDrawable.setColorFilter(cf);
    }

    @Override
    public void setColorFilter(int color, PorterDuff.Mode mode) {
        wrappedDrawable.setColorFilter(color, mode);
    }

    @Override
    public void clearColorFilter() {
        wrappedDrawable.clearColorFilter();
    }

    @Override
    public boolean isStateful() {
        return wrappedDrawable.isStateful();
    }

    @Override
    public boolean setState(int[] stateSet) {
        return wrappedDrawable.setState(stateSet);
    }

    @Override
    public int[] getState() {
        return wrappedDrawable.getState();
    }

    @TargetApi(Build.VERSION_CODES.HONEYCOMB)
    @Override
    public void jumpToCurrentState() {
        wrappedDrawable.jumpToCurrentState();
    }

    @Override
    public Drawable getCurrent() {
        return wrappedDrawable.getCurrent();
    }

    @Override
    public boolean setVisible(boolean visible, boolean restart) {
        return wrappedDrawable.setVisible(visible, restart);
    }

    @Override
    public int getOpacity() {
        return wrappedDrawable.getOpacity();
    }

    @Override
    public Region getTransparentRegion() {
        return wrappedDrawable.getTransparentRegion();
    }

    @Override
    public int getIntrinsicWidth() {
        return wrappedDrawable.getIntrinsicWidth();
    }

    @Override
    public int getIntrinsicHeight() {
        return wrappedDrawable.getIntrinsicHeight();
    }

    @Override
    public int getMinimumWidth() {
        return 0;
    }

    @Override
    public int getMinimumHeight() {
        return 0;
    }

    @Override
    public boolean getPadding(Rect padding) {
        return wrappedDrawable.getPadding(padding);
    }

    @Override
    public Drawable mutate() {
        return this;
    }

    @Override
    public void inflate(Resources r, XmlPullParser parser, AttributeSet attrs) throws XmlPullParserException, IOException {
        wrappedDrawable.inflate(r, parser, attrs);
    }

    @Override
    public ConstantState getConstantState() {
        return wrappedDrawable.getConstantState();
    }
}
