package net.doo.snap.lib.ui;

import android.content.ClipboardManager;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;
import com.commonsware.cwac.camera.CameraFragment;
import com.google.android.gms.maps.CameraUpdateFactory;
import com.google.android.gms.maps.GoogleMap;
import com.google.android.gms.maps.MapFragment;
import com.google.android.gms.maps.model.LatLng;
import com.google.android.gms.maps.model.MarkerOptions;
import com.google.inject.Inject;
import com.google.zxing.Result;
import com.google.zxing.client.result.GeoParsedResult;
import com.google.zxing.client.result.ParsedResultType;
import net.doo.snap.lib.R;
import net.doo.snap.lib.snap.camera.barcode.BarcodeDetector;
import net.doo.snap.lib.snap.camera.barcode.result.ResultHandler;
import net.doo.snap.lib.snap.camera.barcode.result.ResultHandlerFactory;
import net.doo.snap.lib.snap.util.OrientationLocker;
import net.doo.snap.lib.ui.widget.text.CustomTypefaceTextView;
import roboguice.activity.RoboFragmentActivity;

/**
 * Represents barcode content.
 */
public class BarcodeActivity extends RoboFragmentActivity {
    private static final int DEFAULT_MAP_ZOOM = 15;

    @Inject
    private BarcodeDetector barcodeDetector;
    @Inject
    private Context context;
    @Inject
    private OrientationLocker orientationLocker;

    private ImageView codeTypeImage;
    private Result lastCodeResult;
    private TextView title;
    private TextView subTitle;
    private TextView contentText;
    private LinearLayout buttonsContainer;
    private ResultHandler barcodeHandler;
    private WebView webContainer;
    private View contentContainer;
    private GoogleMap map;
    private View mapContainer;
    private CameraFragment cameraFragment;

    @Override
    public void onBackPressed() {
        if (barcodeHandler.getResult().getType() == ParsedResultType.URI && webContainer.canGoBack()) {
            webContainer.goBack();
        } else {
            overridePendingTransition(0, 0);
            super.onBackPressed();
        }
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.barcode_activity);
        lastCodeResult = barcodeDetector.getLastResult();
        if (lastCodeResult == null) {
            finish();
            return;
        }

        barcodeHandler = ResultHandlerFactory.makeResultHandler(this, lastCodeResult);

        getActionBar().setTitle(context.getString(R.string.qr_code_fragment_title));
        getActionBar().setHomeButtonEnabled(true);
        getActionBar().setDisplayHomeAsUpEnabled(true);

        codeTypeImage = (ImageView) findViewById(R.id.type_icon);
        title = (TextView) findViewById(R.id.title);
        subTitle = (TextView) findViewById(R.id.subtitle);
        contentText = (TextView) findViewById(R.id.text);
        contentContainer = findViewById(R.id.content_container);
        buttonsContainer = (LinearLayout) findViewById(R.id.buttons_container);
        webContainer = (WebView) findViewById(R.id.web_container);
        mapContainer = findViewById(R.id.map);
        cameraFragment = (CameraFragment) getSupportFragmentManager().findFragmentById(R.id.camera);

        MapFragment mapFragment = (MapFragment) getFragmentManager().findFragmentById(R.id.map);
        mapFragment.getView().setVisibility(View.GONE);
        map = mapFragment.getMap();
        initContent();
        initActionButton();
    }

    private void initContent() {
        String displayTitle = barcodeHandler.getDisplayTitle();
        title.setVisibility(TextUtils.isEmpty(displayTitle) ? View.GONE : View.VISIBLE);
        title.setText(displayTitle);

        String displaySubtitle = barcodeHandler.getDisplaySubtitle();
        subTitle.setVisibility(TextUtils.isEmpty(displaySubtitle) ? View.GONE : View.VISIBLE);
        subTitle.setText(displaySubtitle);

        String displayText = barcodeHandler.getDisplayText();
        contentText.setVisibility(TextUtils.isEmpty(displayText) ? View.GONE : View.VISIBLE);
        contentText.setText(displayText);

        codeTypeImage.setImageResource(barcodeHandler.getDisplayIconRes());

        switch (barcodeHandler.getResult().getType()) {
            case ADDRESSBOOK:
            case EMAIL_ADDRESS:
            case PRODUCT:
            case WIFI:
            case TEL:
            case SMS:
            case CALENDAR:
            case ISBN:
                break;
            case URI:
                hideCamera();
                contentContainer.setVisibility(View.GONE);
                codeTypeImage.setVisibility(View.GONE);
                webContainer.setVisibility(View.VISIBLE);
                buttonsContainer.setBackgroundColor(getResources().getColor(R.color.qr_code_button_bg));

                webContainer.getSettings().setJavaScriptEnabled(true);
                webContainer.setWebViewClient(new WebViewClient());
                webContainer.loadUrl(barcodeHandler.getDisplayTitle());
                break;
            case GEO:
                if (map == null) {
                    break;
                }
                hideCamera();
                contentContainer.setVisibility(View.GONE);
                codeTypeImage.setVisibility(View.GONE);
                buttonsContainer.setBackgroundColor(getResources().getColor(R.color.qr_code_button_bg));

                mapContainer.setVisibility(View.VISIBLE);
                GeoParsedResult geoResult = (GeoParsedResult) barcodeHandler.getResult();
                LatLng position = new LatLng(geoResult.getLatitude(), geoResult.getLongitude());
                map.addMarker(new MarkerOptions().position(position));

                map.moveCamera(CameraUpdateFactory.newLatLngZoom(position, DEFAULT_MAP_ZOOM));
                break;
            default:
                codeTypeImage.setVisibility(View.GONE);
                contentText.setGravity(Gravity.LEFT);
                contentText.setTextSize(TypedValue.COMPLEX_UNIT_PX,
                        getResources().getDimensionPixelSize(R.dimen.barcode_text_size));
                break;
        }
    }

    private void hideCamera() {
        getSupportFragmentManager().beginTransaction().remove(cameraFragment);
        findViewById(R.id.camera_overlay).setVisibility(View.GONE);
    }

    private void initActionButton() {
        buttonsContainer.setVisibility(barcodeHandler.getButtonCount() > 0 ? View.VISIBLE : View.GONE);
        for (int index = 0; index < barcodeHandler.getButtonCount(); index++) {
            CustomTypefaceTextView button = (CustomTypefaceTextView) getLayoutInflater().inflate(R.layout.barcode_button, null);
            button.setText(barcodeHandler.getButtonText(index));
            final int finalIndex = index;
            button.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    barcodeHandler.handleButtonPress(finalIndex);
                }
            });
            buttonsContainer.addView(button);
        }
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        getMenuInflater().inflate(R.menu.barcode_menu, menu);
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        if (item.getItemId() == R.id.copy) {
            ClipboardManager clipboard = (android.content.ClipboardManager) getSystemService(Context.CLIPBOARD_SERVICE);
            android.content.ClipData clip = android.content.ClipData.newPlainText("", barcodeHandler.getResult().getDisplayResult());
            clipboard.setPrimaryClip(clip);
            Toast.makeText(this, getString(R.string.msg_copied_to_clipboard), Toast.LENGTH_SHORT).show();
            return true;
        } else if (item.getItemId() == R.id.share) {
            Intent sharingIntent = new Intent(android.content.Intent.ACTION_SEND);
            sharingIntent.setType("text/plain");
            sharingIntent.putExtra(android.content.Intent.EXTRA_TEXT, barcodeHandler.getResult().getDisplayResult());
            startActivity(Intent.createChooser(sharingIntent, getString(R.string.share_via_title)));
            return true;
        } else if (item.getItemId() == android.R.id.home) {
            finish();
            return true;
        }

        return super.onOptionsItemSelected(item);
    }

}
