package net.doo.snap.lib.persistence;

import android.app.Application;
import android.os.AsyncTask;
import android.os.Looper;

import com.google.inject.Inject;

import net.doo.snap.lib.util.log.DebugLog;

import org.apache.commons.io.FileUtils;

import java.io.File;
import java.io.IOException;

import static net.doo.snap.lib.util.FileUtils.getExternalFilesDirOrShowError;

/**
 * Provides {@link java.io.File} directories for saving {@link net.doo.snap.lib.persistence.Page}s
 *
 * @see net.doo.snap.lib.persistence.PageFactory
 */
public class PageStoreStrategy {

    private Application context;

    @Inject
    public PageStoreStrategy(Application context) {
        this.context = context;
    }

    /**
     * @return {@link File} directory where {@link net.doo.snap.lib.persistence.Page}s will be stored or {@code null} if not possible to create
     * @throws java.io.IOException if directory can't be opened
     */
    public File getPagesDir() throws IOException {
        return getExternalFilesDirOrShowError(context, "snapping_pages");
    }

    /**
     * @return {@link java.io.File} directory where {@link net.doo.snap.lib.persistence.Page} stored
     * or {@code null} if thehre wre problems accessing FS
     * @throws java.io.IOException if directory can't be opened
     */
    public File getPageDir(String pageId) throws IOException {
        return FileUtils.getFile(getPagesDir(), pageId);
    }

    /**
     * Erase page directory from filesystem
     *
     * @param pageId
     */
    public void erasePage(final String pageId) {
        if (Looper.myLooper() == Looper.getMainLooper()) {
            new AsyncTask<String, Void, Void>() {

                @Override
                protected Void doInBackground(String... params) {
                    String pageId = params[0];
                    erasePageSync(pageId);
                    return null;
                }
            }.execute(pageId);
        } else {
            erasePageSync(pageId);
        }
    }

    /**
     * Erase page directory synchronously
     *
     * @param pageId
     */
    private void erasePageSync(String pageId) {
        try {
            FileUtils.deleteDirectory(getPageDir(pageId));
        } catch (IOException e) {
            DebugLog.logException(e);
        }
    }
}
