package net.doo.snap.lib.util.log;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.os.Environment;
import android.text.TextUtils;
import android.util.Log;

import java.io.Closeable;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Enables logging to system log/sdcard file.
 */
public final class DebugLog {
    public static final String LOG_DIRECTORY = "debug_logs";
    public static final String LOG_EXTENSION = ".txt";
    private static final String LOG_FIELD_SEPARATOR = " | ";
    private static final String UNKNOWN_SIGNATURE = "[unknown]";
    private static final String DEFAULT_PACKAGE_NAME = "default";
    private static final String NO_APPLICATION_INFO_MESSAGE = "[No application info]";
    private static final SimpleDateFormat TIME_FORMAT =
            new SimpleDateFormat("[yyyy-MM-dd HH:mm:ss.SSS] ");
    private static final Pattern CLASS_NAME_PATTERN =
            Pattern.compile("([A-Z]*|(^[a-z]))[_\\da-z\\$]*");
    private static boolean prepared = false;
    private static boolean savingEnabled = true;
    private static boolean echoEnabled = true;
    private static Level savingLevel = Level.Verbose;
    private static Level echoLevel = Level.Verbose;
    private static String packageName = DEFAULT_PACKAGE_NAME;

    /**
     * Logging level.
     */
    public static enum Level {
        Verbose("V"),
        Debug("D"),
        Information("I"),
        Warning("W"),
        Error("E");
        private final String label;

        private Level(String label) {
            this.label = label;
        }

        @Override
        public String toString() {
            return label;
        }
    }

    private DebugLog() {
    }

    public static synchronized void prepare(Context context) {
        prepared = true;

        if (context != null) {
            packageName = context.getPackageName();
            printVersionInfo(context);
        } else {
            packageName = DEFAULT_PACKAGE_NAME;
        }
    }

    private static void printVersionInfo(Context context) {
        PackageManager packageMgr = context.getPackageManager();
        String message;
        try {
            PackageInfo packageInfo = packageMgr.getPackageInfo(packageName, 0);
            int appLabelId = packageInfo.applicationInfo.labelRes;
            message = (appLabelId != 0 ? context.getString(appLabelId) : UNKNOWN_SIGNATURE)
                    + " v" + packageInfo.versionName + " #" + packageInfo.versionCode;
        } catch (NameNotFoundException e) {
            DebugLog.logException(e);
            message = NO_APPLICATION_INFO_MESSAGE;
        }
        DebugLog.i("APP_INFO", message);
    }

    /**
     * Writes tag and message to the log file and also calls Log.d with given params.
     */
    public static void d(String tag, String message) {
        logMessage(tag, message, Level.Debug);
    }

    /**
     * Writes tag and message to the log file and also calls Log.w with given params.
     */
    public static void w(String tag, String message) {
        logMessage(tag, message, Level.Warning);
    }

    /**
     * Writes tag and message to the log file and also calls Log.e with given params.
     */
    public static void e(String tag, String message) {
        logMessage(tag, message, Level.Error);
    }

    /**
     * Writes tag and message to the log file and also calls Log.i with given params.
     */
    public static void i(String tag, String message) {
        logMessage(tag, message, Level.Information);
    }

    /**
     * Writes a message about exception to system log and to the file.
     *
     * @param tag   tag string
     * @param where a string describing where the exception happened
     * @param e     an exception which class name and message will be saved in log
     */
    public static void e(String tag, String where, Throwable e) {
        String message = new StringBuilder("Exception in ")
                .append(where).append(": ")
                .append(e.getClass().getName()).append(LOG_FIELD_SEPARATOR)
                .append(e.getMessage())
                .toString();
        e(tag, message);
    }

    public static void d(String message) {
        String tag = getCallerClassName();
        d(tag, message);
    }

    public static void i(String message) {
        String tag = getCallerClassName();
        i(tag, message);
    }

    public static void w(String message) {
        String tag = getCallerClassName();
        w(tag, message);
    }

    public static void e(String message) {
        String tag = getCallerClassName();
        e(tag, message);
    }

    public static void logException(Throwable e) {
        String tag = getCallerClassName();
        String where = getCallerMethodName();
        if (e != null) {
            e(tag, where, e);
        } else {
            e(tag, where, new Exception("Unknown exception"));
        }
    }

    public static void logExceptionWithStackTrace(Throwable e) {
        String tag = getCallerClassName();
        if (e != null) {
            String exceptionStackTrace = Log.getStackTraceString(e);
            e(tag, exceptionStackTrace);
        } else {
            e(tag, "", new Exception("Unknown exception"));
        }
    }

    public static void logMethod() {
        String tag = getCallerClassName();
        String where = getCallerMethodName();
        i(tag, where);
    }

    public static boolean isSavingEnabled() {
        return savingEnabled;
    }

    /**
     * When not enabled, messages are not stored to the log file.
     */
    public static synchronized void setSavingEnabled(boolean enabled) {
        DebugLog.savingEnabled = enabled;
    }

    public static boolean isEchoEnabled() {
        return echoEnabled;
    }

    /**
     * When echo is not enabled, messages are not printed to Android Log.
     */
    public static void setEchoEnabled(boolean echoEnabled) {
        DebugLog.echoEnabled = echoEnabled;
    }

    public static Level getLevel() {
        return savingLevel;
    }

    public static void setLevel(Level level) {
        if (level != null) {
            DebugLog.savingLevel = level;
        }
    }

    public static Level getEchoLevel() {
        return echoLevel;
    }

    public static void setEchoLevel(Level echoLevel) {
        if (echoLevel != null) {
            DebugLog.echoLevel = echoLevel;
        }
    }

    public static void logStackTrace() {
        String tag = getCallerClassName();
        i(tag, "print stack trace");
        i(tag, "-----------------");
        StackTraceElement[] trace = Thread.currentThread().getStackTrace();
        final int callertStackIndex = 3;
        StringBuilder sb = new StringBuilder();
        for (int i = callertStackIndex; i < trace.length; i++) {
            StackTraceElement stackTraceElement = trace[i];
            sb.append("at ");
            sb.append(stackTraceElement.getClassName());
            sb.append('.');
            sb.append(stackTraceElement.getMethodName());
            sb.append('(');
            if (stackTraceElement.isNativeMethod()) {
                sb.append("Native Method");
            } else {
                sb.append(stackTraceElement.getFileName());
                sb.append(':');
                sb.append(stackTraceElement.getLineNumber());
            }
            sb.append(")\n");
        }
        i(tag, sb.toString());
        i(tag, "-----------------");
    }

    private static void logMessage(String tag, String message, Level level) {
        if (!prepared) {
            return;
        }

        if (echoEnabled && level.ordinal() >= echoLevel.ordinal()) {
            echoMessage(tag, message, level);
        }
        if (savingEnabled && level.ordinal() >= savingLevel.ordinal()) {
            logToFile(tag, message, level);
        }
    }

    private static void echoMessage(String tag, String message, Level level) {
        switch (level) {
            case Debug:
                Log.d(tag, message);
                break;
            case Error:
                Log.e(tag, message);
                break;
            case Warning:
                Log.w(tag, message);
                break;
            case Information:
                Log.i(tag, message);
                break;
            case Verbose:
                Log.v(tag, message);
                break;
        }
    }

    private static synchronized void logToFile(String tag, String message, Level level) {
        Writer writer = getLogWriter();
        if (writer != null) {
            try {
                writer.write(buildLogMessage(tag, message, level));
            } catch (IOException e) {
                // can't do anything about this exception
            } finally {
                closeSafely(writer);
            }
        }
    }

    private static void closeSafely(Closeable closeable) {
        try {
            closeable.close();
        } catch (IOException e) {
            // ignoring
        }
    }

    private static String buildLogMessage(String tag, String message, Level level) {
        String editedTag = tag != null ? tag.replace('|', '/') : "";
        return TIME_FORMAT.format(new Date()) + level + LOG_FIELD_SEPARATOR + editedTag + LOG_FIELD_SEPARATOR + message + "\r\n";
    }

    private static Writer getLogWriter() {
        FileWriter writer = null;
        File externalStorage = Environment.getExternalStorageDirectory();
        prepareLogsFolder(externalStorage);
        File logfile = new File(externalStorage, getLogFilename());
        try {
            writer = new FileWriter(logfile, true);
        } catch (IOException e) {
            // ignoring this
        }
        return writer;
    }

    private static void prepareLogsFolder(File externalStorage) {
        File logDir = new File(externalStorage, LOG_DIRECTORY);
        if (!logDir.exists()) {
            logDir.mkdir();
        }
    }

    private static String getLogFilename() {
        return LOG_DIRECTORY + File.separatorChar + packageName + LOG_EXTENSION;
    }

    private static String tokenizeClassName(String className) {
        List<String> parts = new ArrayList<String>();
        String result;
        try {
            Matcher matcher = CLASS_NAME_PATTERN.matcher(className);
            while (matcher.find()) {
                String part = className.substring(matcher.start(), matcher.end());
                if (!TextUtils.isEmpty(part.trim())) {
                    parts.add(part.toUpperCase());
                }
            }
            result = parts.isEmpty() ? className : TextUtils.join("_", parts);
        } catch (Exception e) {
            result = className;
        }
        return result;
    }

    private static String getCallerClassName() {
        String className;
        final int CALLER_STACK_INDEX = 4;
        try {
            StackTraceElement[] callStack = Thread.currentThread().getStackTrace();
            if (callStack.length > CALLER_STACK_INDEX) {
                String fullClassName = callStack[CALLER_STACK_INDEX].getClassName();
                int lastPoint = fullClassName.lastIndexOf(".");
                if (lastPoint >= 0 && lastPoint + 1 < fullClassName.length()) {
                    className = fullClassName.substring(lastPoint + 1);
                } else {
                    className = UNKNOWN_SIGNATURE;
                }
            } else {
                className = UNKNOWN_SIGNATURE;
            }
        } catch (Exception e) {
            className = UNKNOWN_SIGNATURE;
        }
        return tokenizeClassName(className);
    }

    private static String getCallerMethodName() {
        String methodName;
        final int CALLER_STACK_INDEX = 4;
        try {
            StackTraceElement[] callStack = Thread.currentThread().getStackTrace();
            if (callStack.length > CALLER_STACK_INDEX) {
                methodName = callStack[CALLER_STACK_INDEX].getMethodName();
            } else {
                methodName = UNKNOWN_SIGNATURE;
            }
        } catch (Exception e) {
            methodName = UNKNOWN_SIGNATURE;
        }
        return methodName;
    }

    private static String getThreadName() {
        return Thread.currentThread().getName();
    }
}
