package net.doo.snap.lib.analytics;

import android.app.Activity;

import com.adjust.sdk.Adjust;
import com.google.analytics.tracking.android.EasyTracker;

import net.doo.snap.lib.BuildConfig;
import net.doo.snap.lib.util.log.DebugLog;
import org.apache.commons.lang.StringUtils;

import java.util.HashMap;
import java.util.Map;

/**
 * Wraps {@link com.google.analytics.tracking.android.EasyTracker} to not produce analytic reports while
 * in debug mode
 */
public class EasyTrackerWrapper {

    private static final String ANALYTICS_EVENT_TAG = "ANALYTICS_EVENT";

    /**
     * @see com.google.analytics.tracking.android.Tracker#sendEvent(String, String, String, Long)
     */
    public static void sendEvent(String category, String action, String label, Long value) {
        DebugLog.d(ANALYTICS_EVENT_TAG, StringUtils.join(new String[]{category, action, label, String.valueOf(value)}, "->"));

        if (BuildConfig.DEBUG) return;
        EasyTracker.getTracker().sendEvent(category, action, label, value);

        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put("category", category);
        parameters.put("action", action);
        parameters.put("label", label);
        parameters.put("value", String.valueOf(value));
        Adjust.trackEvent(AnalyticsConst.ADJUST_ANDROID_EVENT_TOKEN, parameters);
    }

    /**
     * @see com.google.analytics.tracking.android.EasyTracker#activityStart(android.app.Activity)
     */
    public static void activityStart(Activity activity) {
        DebugLog.d(ANALYTICS_EVENT_TAG, "Activity START " + activity.getClass().toString());

        if (BuildConfig.DEBUG) return;
        EasyTracker.getInstance().activityStart(activity);
    }

    /**
     * @see com.google.analytics.tracking.android.EasyTracker#activityStop(android.app.Activity)
     */
    public static void activityStop(Activity activity) {
        DebugLog.d(ANALYTICS_EVENT_TAG, "Activity STOP " + activity.getClass().toString());

        if (BuildConfig.DEBUG) return;
        EasyTracker.getInstance().activityStop(activity);
    }

    /**
     * @see com.adjust.sdk.Adjust#onResume(android.app.Activity)
     */
    public static void activityResume(Activity activity) {
        DebugLog.d(ANALYTICS_EVENT_TAG, "Activity RESUME " + activity.getClass().toString());

        if (BuildConfig.DEBUG) return;
        Adjust.onResume(activity);
    }

    /**
     * @see com.adjust.sdk.Adjust#onPause()
     */
    public static void activityPause(Activity activity) {
        DebugLog.d(ANALYTICS_EVENT_TAG, "Activity PAUSE " + activity.getClass().toString());

        if (BuildConfig.DEBUG) return;
        Adjust.onPause();
    }

}
