package net.doo.snap.lib.billing;

import android.os.Handler;
import android.os.Looper;

import java.util.concurrent.Executor;

/**
 * Encapsulates {@link net.doo.snap.lib.billing.Feature} execution and corresponding error handling.
 */
public class FeatureExecutor {

    private final BillingManager billingManager;
    private ResolutionStrategy resolutionStrategy;
    private Executor executor;

    private Handler mainThreadHandler = new Handler(Looper.getMainLooper());

    public FeatureExecutor(BillingManager billingManager) {
        this.billingManager = billingManager;
    }

    /**
     * @return new {@link net.doo.snap.lib.billing.FeatureExecutor}
     */
    public static FeatureExecutor build(BillingManager billingManager) {
        return new FeatureExecutor(billingManager);
    }

    /**
     * Assigns {@link net.doo.snap.lib.billing.ResolutionStrategy} which will be used in case if {@link net.doo.snap.lib.billing.Feature}
     * is not available. {@link net.doo.snap.lib.billing.ResolutionStrategy#resolveNotAvailableFeature(Feature)} is guaranteed to be invoked
     * in main thread.
     *
     * @return same {@link net.doo.snap.lib.billing.FeatureExecutor}
     */
    public FeatureExecutor setResolutionStrategy(ResolutionStrategy resolutionStrategy) {
        this.resolutionStrategy = resolutionStrategy;

        return this;
    }

    /**
     * Assigns {@link java.util.concurrent.Executor} which will be used for blocking requests to {@link net.doo.snap.lib.billing.BillingManager}.
     * Executor must be fully prepared for execution of {@link java.lang.Runnable}s
     *
     * @return same {@link net.doo.snap.lib.billing.FeatureExecutor}.
     */
    public FeatureExecutor setExecutor(Executor executor) {
        this.executor = executor;

        return this;
    }

    /**
     * Tries to execute provided {@link net.doo.snap.lib.billing.Feature}. This might be performed instantly,
     * or might initiate execution on {@link java.util.concurrent.Executor}. Users of this method should not made any assumptions on whether
     * {@link net.doo.snap.lib.billing.Feature} will be executed instantly or not.
     * <p/>
     * If {@link net.doo.snap.lib.billing.Feature} is not available for the user, {@link net.doo.snap.lib.billing.FeatureExecutor}
     * will first refresh {@link net.doo.snap.lib.billing.BillingManager} and then (if execution still fails)
     * invoke provided (if any) {@link net.doo.snap.lib.billing.ResolutionStrategy}.
     */
    public void executeFeature(Feature feature) {
        if (billingManager.isItemAvailable(feature.getId())) {
            feature.run();
        } else if (billingManager.isInitialized() && executor != null) {
            refreshItemsAndRetry(feature);
        }
    }

    private void refreshItemsAndRetry(final Feature feature) {
        executor.execute(new Runnable() {
            @Override
            public void run() {
                if (!billingManager.isItemAvailable(feature.getId())) {
                    billingManager.refreshItems();
                }

                mainThreadHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        retryFeatureExecution(feature);
                    }
                });
            }
        });
    }

    private void retryFeatureExecution(Feature feature) {
        if (billingManager.isItemAvailable(feature.getId())) {
            feature.run();
        } else if (resolutionStrategy != null) {
            resolutionStrategy.resolveNotAvailableFeature(feature);
        }
    }

}
