package net.doo.snap.lib.ui.widget.text.processor;

import android.text.SpannableStringBuilder;

import java.util.ArrayList;

/**
 * Processes raw {@link java.lang.String} input and converts it to formatted {@link java.lang.CharSequence}.
 * <p/>
 * Formatting process consists of a single iteration through an input. For each character {@link net.doo.snap.lib.ui.widget.text.processor.TextProcessor}
 * tries to invoke a {@link net.doo.snap.lib.ui.widget.text.processor.TextProcessor.ProcessStrategy} which might either skip or consume character.
 * If character was consumed, then no other {@link net.doo.snap.lib.ui.widget.text.processor.TextProcessor.ProcessStrategy} will be invoked
 * as long as current one returns {@code true}.
 * <p/>
 * If character wasn't processed by any {@link net.doo.snap.lib.ui.widget.text.processor.TextProcessor.ProcessStrategy} then it's added
 * to output as is. Strategies are responsible for appending characters to output if they're consumed.
 */
public class TextProcessor {

    /**
     * Entity which might perform input modification. Strategy will be provided with each non-consumed character
     * of the input which it might then either skip (and it will be passed to next strategy or, if no one will consume it,
     * will be added to output as is) or consume. Consumed characters are not added to output automatically - strategy should
     * perform that if needed.
     * <p/>
     * As soon as strategy will consume character it will block all other strategies from receiving characters as long as it keep
     * consuming further characters.
     */
    public interface ProcessStrategy {

        /**
         * @param character {@code char} to process
         * @param output    mutable output ot {@link net.doo.snap.lib.ui.widget.text.processor.TextProcessor}
         * @return {@code true} if character was consumed. {@code false} if it was skipped.
         */
        boolean processChar(char character, SpannableStringBuilder output);

        /**
         * Brings strategy to default state, so it should be ready for processing of new input.
         */
        void reset();

    }

    private ArrayList<ProcessStrategy> strategies = new ArrayList<ProcessStrategy>();
    private ProcessStrategy currentStrategy;

    /**
     * Adds new {@link net.doo.snap.lib.ui.widget.text.processor.TextProcessor.ProcessStrategy} to the end of the
     * list.
     *
     * @return same {@link net.doo.snap.lib.ui.widget.text.processor.TextProcessor}
     * @throws NullPointerException if strategy is {@code null}
     */
    public TextProcessor addStrategy(ProcessStrategy strategy) throws NullPointerException {
        if (strategy == null) {
            throw new NullPointerException();
        }

        strategies.add(strategy);
        return this;
    }

    /**
     * @return {@link java.lang.CharSequence} representation of input, formatted according to the rules.
     */
    public CharSequence processText(String input) {
        if (strategies.isEmpty()) {
            return input;
        }

        resetStrategies();

        SpannableStringBuilder output = new SpannableStringBuilder();

        final int strategiesCount = strategies.size();
        final int inputLength = input.length();
        for (int position = 0; position < inputLength; position++) {
            final char currentChar = input.charAt(position);

            ProcessStrategy lastStrategy = null;
            if (currentStrategy != null) {
                if (currentStrategy.processChar(currentChar, output)) {
                    continue;
                } else {
                    lastStrategy = currentStrategy;
                    currentStrategy = null;
                }
            }

            boolean consumed = false;
            for (int i = 0; i < strategiesCount; i++) {
                ProcessStrategy strategy = strategies.get(i);
                if (lastStrategy == strategy) {
                    continue;
                }

                consumed = strategy.processChar(currentChar, output);

                if (consumed) {
                    currentStrategy = strategy;
                    break;
                }
            }

            if (!consumed) {
                output.append(currentChar);
            }
        }

        return output;
    }

    private void resetStrategies() {
        for (ProcessStrategy strategy : strategies) {
            strategy.reset();
        }
    }

}
