package net.doo.snap.lib.util;

/**
 * Holds current {@link net.doo.snap.lib.util.State} and responsible for
 * {@link net.doo.snap.lib.util.State} lifecycle methods invocations.
 * Not thread safe.
 */
public class StateHolder {

    private State currentState;
    private State pendingState;

    /**
     * Creates new {@link net.doo.snap.lib.util.StateHolder} with {@link State#DEFAULT} state
     */
    public StateHolder() {
        this(State.DEFAULT);
    }

    /**
     * @throws NullPointerException if parameter is {@code null}
     */
    public StateHolder(State currentState) throws NullPointerException {
        if (currentState == null) {
            throw new NullPointerException("State cannot be null");
        }

        this.currentState = currentState;
    }

    /**
     * Updates information about current state. If new state equals to current state,
     * then no change is performed and no lifecycle methods invoked
     *
     * @param newState
     * @throws java.lang.NullPointerException if parameter is {@code null}
     */
    public void updateState(State newState) throws NullPointerException {
        if (newState == null) {
            throw new NullPointerException("State cannot be null");
        }

        if (currentState.equals(newState)) {
            return;
        }

        pendingState = newState;

        currentState.onLeave();
        newState.onEnter();

        currentState = newState;
        pendingState = null;

        State jumpTarget = newState.takeReferencedState();
        if (jumpTarget != null) {
            updateState(jumpTarget);
        }
    }

    /**
     * @return current state
     */
    public State getCurrentState() {
        return currentState;
    }

    /**
     * @return state that awaits {@link net.doo.snap.lib.util.State#onEnter()} execution to become
     * a current state of {@link net.doo.snap.lib.util.StateHolder}
     */
    public State getPendingState() {
        return pendingState;
    }
}
