package net.doo.snap.lib.snap;

import android.animation.ArgbEvaluator;
import android.animation.ValueAnimator;
import android.annotation.TargetApi;
import android.app.Activity;
import android.os.Build;
import android.os.Bundle;
import android.view.DragEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import com.google.inject.Inject;

import net.doo.snap.lib.R;
import net.doo.snap.lib.snap.edit.drag.DragLocalState;
import net.doo.snap.lib.snap.edit.events.RetakePageEvent;
import net.doo.snap.lib.util.ui.ViewUtils;

import roboguice.event.EventManager;
import roboguice.fragment.RoboFragment;

/**
 * Encapsulates logic of "Retake photo" drop target: animates according to user gestures and
 * fires {@link net.doo.snap.lib.snap.edit.events.RetakePageEvent} on corresponding drop event
 */
public class RetakePhotoDropTargetFragment extends RoboFragment {

    private static final float DRAG_ENTER_SCALE = 1.15f;
    private static final long DRAG_ENTER_DURATION_MILLIS = 200L;
    private static final ArgbEvaluator ARGB_EVALUATOR = new ArgbEvaluator();

    @Inject
    private EventManager eventManager;

    private int defaultColor;
    private int activeColor;

    private View dropTarget;

    @Override
    public void onAttach(Activity activity) {
        super.onAttach(activity);

        defaultColor = getResources().getColor(R.color.retake_photo_drop_target_bg);
        activeColor = getResources().getColor(R.color.retake_photo_drop_target_hovered_bg);
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        View rootView = inflater.inflate(R.layout.retake_photo_drop_target, container, false);

        dropTarget = rootView.findViewById(R.id.drop_target);

        setupDragListener();

        return rootView;
    }

    @Override
    public void onViewCreated(View view, Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        getView().setAlpha(ViewUtils.ALPHA_TRANSPARENT);
    }

    @TargetApi(Build.VERSION_CODES.HONEYCOMB)
    private void setupDragListener() {
        class DragListener implements View.OnDragListener {

            @Override
            public boolean onDrag(View v, DragEvent event) {
                switch (event.getAction()) {
                    case DragEvent.ACTION_DRAG_STARTED:
                        showDropTarget();
                        break;
                    case DragEvent.ACTION_DRAG_ENTERED:
                        animateDragEnter();
                        animateBackgroundDragEnter();
                        break;
                    case DragEvent.ACTION_DRAG_EXITED:
                        animateDragExit();
                        animateBackgroundDragExit();
                        break;
                    case DragEvent.ACTION_DROP:
                        retakePhoto(
                                (DragLocalState) event.getLocalState(),
                                event.getX(),
                                event.getY()
                        );
                        break;
                    case DragEvent.ACTION_DRAG_ENDED:
                        hideDropTarget();
                        animateDragExit();
                        dropTarget.setBackgroundColor(defaultColor);
                        break;
                }

                return true;
            }
        }

        dropTarget.setOnDragListener(new DragListener());
    }

    private void retakePhoto(DragLocalState localState, float x, float y) {
        eventManager.fire(new RetakePageEvent(
                localState.getPagePosition(),
                x, y
        ));
    }

    private void animateDragEnter() {
        getView().animate()
                .scaleX(DRAG_ENTER_SCALE)
                .scaleY(DRAG_ENTER_SCALE)
                .setDuration(DRAG_ENTER_DURATION_MILLIS)
                .translationY(dropTarget.getHeight() / 2f * (DRAG_ENTER_SCALE - ViewUtils.SCALE_DEFAULT))
                .start();
    }

    private void animateBackgroundDragEnter() {
        ValueAnimator colorAnimator = ValueAnimator.ofObject(
                ARGB_EVALUATOR,
                defaultColor,
                activeColor
        );
        colorAnimator.addUpdateListener(new ColorUpdateListener());
        colorAnimator.start();
    }

    private void animateDragExit() {
        getView().animate()
                .scaleX(ViewUtils.SCALE_DEFAULT)
                .scaleY(ViewUtils.SCALE_DEFAULT)
                .translationY(ViewUtils.TRANSLATION_DEFAULT)
                .start();
    }

    private void animateBackgroundDragExit() {
        ValueAnimator colorAnimator = ValueAnimator.ofObject(
                ARGB_EVALUATOR,
                activeColor,
                defaultColor
        );
        colorAnimator.addUpdateListener(new ColorUpdateListener());
        colorAnimator.start();
    }

    private void showDropTarget() {
        getView().setAlpha(ViewUtils.ALPHA_DEFAULT);
    }

    private void hideDropTarget() {
        getView().animate().alpha(ViewUtils.ALPHA_TRANSPARENT).start();
    }

    /**
     * Updates background color of drop target
     */
    private class ColorUpdateListener implements ValueAnimator.AnimatorUpdateListener {

        @Override
        public void onAnimationUpdate(ValueAnimator animation) {
            dropTarget.setBackgroundColor((Integer) animation.getAnimatedValue());
        }
    }
}
