package net.doo.snap.lib.analytics;

import android.app.Activity;

import com.adjust.sdk.Adjust;
import com.google.android.gms.analytics.GoogleAnalytics;
import com.google.android.gms.analytics.HitBuilders;
import com.google.android.gms.analytics.Tracker;

import net.doo.snap.lib.BuildConfig;
import net.doo.snap.lib.util.log.DebugLog;

import org.apache.commons.lang.StringUtils;

import java.util.HashMap;
import java.util.Map;

/**
 * Wraps {@link com.google.android.gms.analytics.Tracker} to not produce analytic reports while
 * in debug mode
 */
public class EasyTrackerWrapper {

    private static final String ANALYTICS_EVENT_TAG = "ANALYTICS_EVENT";

    private static GoogleAnalytics analytics;
    private static Tracker appTracker;

    /**
     * Initializes wrapper with given {@link com.google.android.gms.analytics.Tracker}
     */
    public static void setAppTracker(GoogleAnalytics analytics, Tracker tracker) {
        EasyTrackerWrapper.analytics = analytics;
        appTracker = tracker;
    }

    /**
     * Sends event to analytics
     */
    public static void sendEvent(String category, String action, String label, Long value) {
        DebugLog.d(ANALYTICS_EVENT_TAG, StringUtils.join(new String[]{category, action, label, String.valueOf(value)}, "->"));

        if (BuildConfig.DEBUG) return;

        appTracker.send(
                new HitBuilders.EventBuilder()
                        .setCategory(category)
                        .setAction(action)
                        .setLabel(label)
                        .setValue(value != null ? value : 0L)
                        .build()
        );

        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put("category", category);
        parameters.put("action", action);
        parameters.put("label", label);
        parameters.put("value", String.valueOf(value));
        Adjust.trackEvent(AnalyticsConst.ADJUST_ANDROID_EVENT_TOKEN, parameters);
    }

    /**
     * Tracks start of {@link android.app.Activity}
     */
    public static void activityStart(Activity activity) {
        DebugLog.d(ANALYTICS_EVENT_TAG, "Activity START " + activity.getClass().toString());

        if (BuildConfig.DEBUG) return;
        analytics.reportActivityStart(activity);
    }

    /**
     * Tracks stop of {@link android.app.Activity}
     */
    public static void activityStop(Activity activity) {
        DebugLog.d(ANALYTICS_EVENT_TAG, "Activity STOP " + activity.getClass().toString());

        if (BuildConfig.DEBUG) return;
        analytics.reportActivityStop(activity);
    }

    /**
     * @see com.adjust.sdk.Adjust#onResume(android.app.Activity)
     */
    public static void activityResume(Activity activity) {
        DebugLog.d(ANALYTICS_EVENT_TAG, "Activity RESUME " + activity.getClass().toString());

        if (BuildConfig.DEBUG) return;
        Adjust.onResume(activity);
    }

    /**
     * @see com.adjust.sdk.Adjust#onPause()
     */
    public static void activityPause(Activity activity) {
        DebugLog.d(ANALYTICS_EVENT_TAG, "Activity PAUSE " + activity.getClass().toString());

        if (BuildConfig.DEBUG) return;
        Adjust.onPause();
    }

    /**
     * Tracks timing
     */
    public static void sendTiming(String category, long durationMillis, String name, String label) {
        if (BuildConfig.DEBUG) return;

        appTracker.send(
                new HitBuilders.TimingBuilder()
                        .setCategory(category)
                        .setValue(durationMillis)
                        .setVariable(name)
                        .setLabel(label)
                        .build()
        );
    }

}
