package net.doo.snap.lib.smartname;

import android.app.Activity;
import android.content.Context;
import android.content.IntentSender;
import android.location.Address;
import android.location.Criteria;
import android.location.Geocoder;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Bundle;
import android.text.TextUtils;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GooglePlayServicesClient;
import com.google.android.gms.common.GooglePlayServicesUtil;
import com.google.android.gms.location.LocationClient;
import com.google.android.gms.location.LocationRequest;
import com.google.inject.Inject;
import net.doo.snap.lib.R;
import net.doo.snap.lib.util.log.DebugLog;
import roboguice.activity.event.OnStartEvent;
import roboguice.activity.event.OnStopEvent;
import roboguice.event.Observes;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Locale;

/**
 * Provides actual {@link net.doo.snap.lib.smartname.TermType} values.
 */
public class SmartNameValuesProvider implements
        GooglePlayServicesClient.ConnectionCallbacks,
        GooglePlayServicesClient.OnConnectionFailedListener {

    private final static int
            CONNECTION_FAILURE_RESOLUTION_REQUEST = 9000;
    private static final int LOCATION_UPDATE_FASTEST_INTERVAL_MILLIS = 1000;
    private static final int LOCATION_UPDATE_INTERVAL_MILLIS = 60000;
    private static final int MAX_RESULTS_COUNT = 1;

    private Activity activity;
    private String cityName;
    private Location location;
    private LocationClient locationClient;
    private LocationManager locationManager;

    private com.google.android.gms.location.LocationListener locationListener = new com.google.android.gms.location.LocationListener() {
        @Override
        public void onLocationChanged(Location location) {
            SmartNameValuesProvider.this.location = location;
            updateCurrentCityName();
            disconnectPlayServices();
        }
    };

    private LocationListener androidLocationListener = new LocationListener() {
        @Override
        public void onLocationChanged(Location location) {
            SmartNameValuesProvider.this.location = location;
            updateCurrentCityName();
        }

        @Override
        public void onStatusChanged(String provider, int status, Bundle extras) {}
        @Override
        public void onProviderEnabled(String provider) {}
        @Override
        public void onProviderDisabled(String provider) {}
    };

    @Inject
    public SmartNameValuesProvider(Activity activity) {
        this.activity = activity;
        cityName = activity.getString(R.string.default_city_name);

        if (GooglePlayServicesUtil.isGooglePlayServicesAvailable(activity) != com.google.android.gms.common.ConnectionResult.SUCCESS) {
            locationManager = (LocationManager) activity.getSystemService(Context.LOCATION_SERVICE);
            Criteria criteria = new Criteria();
            criteria.setAccuracy(Criteria.ACCURACY_LOW);
            location = locationManager.getLastKnownLocation(locationManager.getBestProvider(criteria, false));
            if (location == null) {
                locationManager.requestSingleUpdate(locationManager.getBestProvider(criteria, true), androidLocationListener, null);
            } else {
                updateCurrentCityName();
            }
        }

        locationClient = new LocationClient(activity, this, this);
    }

    @SuppressWarnings("unused")
    public void onStart(@Observes OnStartEvent event) {
        locationClient.connect();
    }

    @SuppressWarnings("unused")
    public void onStop(@Observes OnStopEvent event) {
        if (locationManager != null) {
            locationManager.removeUpdates(androidLocationListener);
        }

        disconnectPlayServices();
    }

    private void disconnectPlayServices() {
        if (locationClient.isConnected()) {
            locationClient.removeLocationUpdates(locationListener);
        }
        locationClient.disconnect();
    }

    /**
     * @param termType
     * @return {@link net.doo.snap.lib.smartname.TermType} value
     */
    public String getTermValue(TermType termType) {
        switch (termType) {
            case CITY:
                return TextUtils.isEmpty(cityName) ? activity.getString(R.string.default_city_name) : cityName;
            case YEAR:
                SimpleDateFormat sdf = new SimpleDateFormat("yyyy");
                return sdf.format(new Date());
            case YEAR_SHORT:
                sdf = new SimpleDateFormat("yy");
                return sdf.format(new Date());
            case MONTH:
                sdf = new SimpleDateFormat("MM");
                return sdf.format(new Date());
            case DAY:
                sdf = new SimpleDateFormat("dd");
                return sdf.format(new Date());
            case HOURS:
                sdf = new SimpleDateFormat("HH");
                return sdf.format(new Date());
            case MINUTES:
                sdf = new SimpleDateFormat("mm");
                return sdf.format(new Date());
            case SECONDS:
                sdf = new SimpleDateFormat("ss");
                return sdf.format(new Date());
            case DEVICE_MODEL:
                return TextUtils.isEmpty(Build.MODEL) ? "" : Build.MODEL;
        }
        return "";
    }

    private void updateCurrentCityName() {
        new AsyncTask<Void, Void, String>() {
            @Override
            protected String doInBackground(Void... params) {
                String cityName = activity.getString(R.string.default_city_name);
                if (location == null) {
                    return cityName;
                }
                Geocoder gcd = new Geocoder(activity, Locale.getDefault());
                try {
                    List<Address> addresses = gcd.getFromLocation(location.getLatitude(), location.getLongitude(), MAX_RESULTS_COUNT);
                    if (addresses.size() > 0) {
                        cityName = addresses.get(0).getLocality();
                    }
                } catch (IOException e) {
                    DebugLog.logException(e);
                }

                return cityName;
            }

            @Override
            protected void onPostExecute(String result) {
                cityName = result;
            }
        }.executeOnExecutor(AsyncTask.SERIAL_EXECUTOR);
    }

    @Override
    public void onConnected(Bundle bundle) {
        DebugLog.d("GooglePlay service connected");
        location = locationClient.getLastLocation();
        updateCurrentCityName();
        requestLocationUpdates();
    }

    private void requestLocationUpdates() {
        LocationRequest request = LocationRequest.create();
        request.setPriority(LocationRequest.PRIORITY_BALANCED_POWER_ACCURACY);
        request.setFastestInterval(LOCATION_UPDATE_FASTEST_INTERVAL_MILLIS);
        request.setInterval(LOCATION_UPDATE_INTERVAL_MILLIS);
        locationClient.requestLocationUpdates(request, locationListener);
    }

    @Override
    public void onDisconnected() {
        DebugLog.d("GooglePlay service disconnected");
    }

    @Override
    public void onConnectionFailed(ConnectionResult connectionResult) {
        if (connectionResult.hasResolution()) {
            try {
                connectionResult.startResolutionForResult(
                        activity,
                        CONNECTION_FAILURE_RESOLUTION_REQUEST);
            } catch (IntentSender.SendIntentException e) {
                DebugLog.logException(e);
            }
        } else {
            DebugLog.e("GooglePlay service connection failed: " + connectionResult.getErrorCode());
        }
    }

}
