package net.doo.snap.lib.snap.edit.drag;

import android.animation.Animator;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.os.Handler;
import android.view.DragEvent;
import android.view.View;

import com.google.inject.Inject;

import net.doo.snap.lib.R;
import net.doo.snap.lib.persistence.Page;
import net.doo.snap.lib.snap.preview.ImagePreviewFragment;
import net.doo.snap.lib.snap.PreviewFragmentPagerAdapter;
import net.doo.snap.lib.snap.edit.EditLock;
import net.doo.snap.lib.snap.edit.PagesEditor;
import net.doo.snap.lib.snap.edit.util.PageEditorHelper;
import net.doo.snap.lib.util.ui.ViewUtils;
import net.doo.snap.lib.ui.widget.ViewPager;

/**
 * Handles pages drag and rearrangement
 */
public class PagesRearranger implements PagesEditor, View.OnDragListener {

    public static final float DRAG_SHADOW_SCALE = 0.5f;
    public static final float DRAG_SHADOW_ALPHA = 0.5f;

    private static final long DRAG_SHADOW_SCALE_DURATION_MILLIS = 150L;

    private static final long SCROLL_DELAY_MILLIS = 500L;

    private static final int SCROLL_LEFT_MODIFIER = -1;
    private static final int SCROLL_RIGHT_MODIFIER = 1;

    private Resources resources;
    private EditLock editLock;

    private float scrollAreaWidth;

    private ViewPager viewPager;
    private PreviewFragmentPagerAdapter adapter;
    private PageEditorHelper editorHelper;

    private Handler pagesScrollHandler = new Handler();

    private boolean waitingForScroll = false;
    private boolean rearrangementIdle = false;
    private boolean rearrangementEndDelayed = false;
    private boolean dragActive = false;
    private boolean dropped = false;

    private DragInfo dragInfo;

    @Inject
    public PagesRearranger(Resources resources, EditLock editLock) {
        this.resources = resources;
        this.editLock = editLock;
    }

    @Override
    public void attachViewPager(ViewPager viewPager, PreviewFragmentPagerAdapter adapter) {
        initParams();

        this.viewPager = viewPager;
        this.adapter = adapter;
        editorHelper = new PageEditorHelper(viewPager, adapter);

        viewPager.setOnDragListener(this);
    }

    private void initParams() {
        scrollAreaWidth = resources.getDimension(R.dimen.pager_scroll_area_width);
    }

    /**
     * Starts rearrangement of {@link net.doo.snap.lib.persistence.Page}
     *
     * @param startX          initial X position for drag
     * @param startY          initial Y position for drag
     * @param view            {@link android.view.View} which will be used to draw drag shadow
     * @param draggedPage     {@link net.doo.snap.lib.persistence.Page} being dragged
     * @param pagePosition    position of {@link net.doo.snap.lib.persistence.Page} being dragged
     * @param draggedFragment {@link net.doo.snap.lib.snap.preview.ImagePreviewFragment} being dragged
     * @param preservedBitmap preserved {@link android.graphics.Bitmap} which might be used to display {@link net.doo.snap.lib.persistence.Page}
     *                        right after drop-in event, which lets to avoid unnecessary image loading.
     */
    public void startRearrangement(float startX,
                                   float startY,
                                   View view,
                                   Page draggedPage,
                                   int pagePosition,
                                   ImagePreviewFragment draggedFragment,
                                   Bitmap preservedBitmap) {

        final DragLocalState dragLocalState = new DragLocalState(
                draggedFragment,
                preservedBitmap,
                draggedPage,
                pagePosition
        );

        final View viewToAnimate = view;

        viewToAnimate.setOnDragListener(new View.OnDragListener() {
            @Override
            public boolean onDrag(View v, DragEvent event) {
                if (event.getAction() == DragEvent.ACTION_DRAG_ENDED) {
                    ViewUtils.restoreViewTransformation(viewToAnimate);
                }
                return false;
            }
        });

        editLock.lockEdit();
        ViewUtils.saveViewTransformation(viewToAnimate);
        viewToAnimate.animate()
                .scaleX(DRAG_SHADOW_SCALE)
                .scaleY(DRAG_SHADOW_SCALE)
                .translationX(startX - viewToAnimate.getWidth() / 2f)
                .translationY(startY - viewToAnimate.getHeight() / 2f)
                .alpha(DRAG_SHADOW_ALPHA)
                .setDuration(DRAG_SHADOW_SCALE_DURATION_MILLIS)
                .setListener(new ViewUtils.DefaultAnimationListener() {
                    @Override
                    public void onAnimationFinished(Animator animation) {
                        boolean dragStarted = viewToAnimate.startDrag(
                                null,
                                new PageShadowBuilder(viewToAnimate),
                                dragLocalState,
                                0
                        );

                        if (!dragStarted) {
                            editLock.unlockEdit();
                            ViewUtils.restoreViewTransformation(viewToAnimate);
                        } else {
                            viewToAnimate.setAlpha(ViewUtils.ALPHA_TRANSPARENT);
                        }
                    }
                })
                .start();
    }

    @Override
    public boolean onDrag(View view, DragEvent dragEvent) {
        ImagePreviewFragment currentFragment = adapter.getFragment(viewPager, viewPager.getCurrentItem());

        switch (dragEvent.getAction()) {
            case DragEvent.ACTION_DRAG_STARTED:
                if (adapter.getPagesCount() == 1) {
                    return false;
                }

                dragInfo = new DragInfo((DragLocalState) dragEvent.getLocalState());

                onRearrangementStarted();
                dragActive = true;
                dropped = false;
                break;
            case DragEvent.ACTION_DRAG_LOCATION:
                if (currentFragment == null
                        || dragInfo == null
                        || dragInfo.getDragLocalState() == null
                        || currentFragment.equals(dragInfo.getDragLocalState().getDraggedFragment())) {
                    break;
                }

                if (handleScroll(dragEvent)) {
                    currentFragment.hideDropTarget();
                    break;
                }

                if (dragEvent.getX() < viewPager.getWidth() / 2f) {
                    currentFragment.showLeftDropTarget();
                } else {
                    currentFragment.showRightDropTarget();
                }

                break;
            case DragEvent.ACTION_DRAG_EXITED:
                if (currentFragment != null) {
                    currentFragment.hideDropTarget();
                }

                break;
            case DragEvent.ACTION_DROP:
                dragInfo.setDropPosition(dragEvent.getX(), dragEvent.getY());

                if (!rearrangementIdle) {
                    rearrangementEndDelayed = true;
                    break;
                }

                performDropAction();
                break;
            case DragEvent.ACTION_DRAG_ENDED:
                if (dragActive && !dropped && !rearrangementEndDelayed) {
                    cancelRearrangement(dragEvent);
                }
                currentFragment.hideDropTarget();
                dragActive = false;
                editLock.unlockEdit();
                break;
        }

        return true;
    }

    private void cancelRearrangement(DragEvent dragEvent) {
        try {
            DragLocalState dragLocalState = (DragLocalState) dragEvent.getLocalState();
            adapter.addPage(dragLocalState.getPagePosition(), dragLocalState.getPage());
            viewPager.setCurrentItem(dragLocalState.getPagePosition());
        } finally {
            finalizeRearrangement();
        }
    }

    private void performDropAction() {
        ImagePreviewFragment currentFragment = adapter.getFragment(viewPager, viewPager.getCurrentItem());

        stopScroll();
        currentFragment.hideDropTarget();
        onRearrangementEnded();
    }

    private boolean handleScroll(DragEvent dragEvent) {
        final int currentItem = viewPager.getCurrentItem();
        final float x = dragEvent.getX();

        if (currentItem > 0 && x < scrollAreaWidth) {
            if (!waitingForScroll) {
                pagesScrollHandler.postDelayed(
                        new ScrollRunnable(SCROLL_LEFT_MODIFIER),
                        SCROLL_DELAY_MILLIS
                );
            }

            waitingForScroll = true;
            return true;
        }

        if (currentItem < adapter.getPagesCount() - 1
                && x > viewPager.getWidth() - scrollAreaWidth) {
            if (!waitingForScroll) {
                pagesScrollHandler.postDelayed(
                        new ScrollRunnable(SCROLL_RIGHT_MODIFIER),
                        SCROLL_DELAY_MILLIS
                );
            }

            waitingForScroll = true;
            return true;
        }

        stopScroll();

        return false;
    }

    private void stopScroll() {
        pagesScrollHandler.removeCallbacksAndMessages(null);
        waitingForScroll = false;
    }

    private void onRearrangementStarted() {
        rearrangementIdle = false;

        editorHelper.deleteCurrentPage(new ViewUtils.DefaultAnimationListener() {
            @Override
            public void onAnimationFinished(Animator animation) {
                viewPager.post(new Runnable() {
                    @Override
                    public void run() {
                        onRearrangementIdle();
                    }
                });
            }
        });
    }

    private void onRearrangementIdle() {
        rearrangementIdle = true;

        if (rearrangementEndDelayed) {
            performDropAction();
            rearrangementEndDelayed = false;
        }
    }

    private void onRearrangementEnded() {
        try {
            final Page page = dragInfo.getDragLocalState().getPage();

            page.getParameters().putBoolean(ImagePreviewFragment.PLAY_DROP_IN_ANIMATION, true);
            page.getParameters().putFloat(ImagePreviewFragment.DROP_X, dragInfo.getDropPositionX());
            page.getParameters().putFloat(ImagePreviewFragment.DROP_Y, dragInfo.getDropPositionY());

            Bitmap preservedBitmap = dragInfo.getDragLocalState().getPreservedBitmap();
            if (preservedBitmap != null) {
                page.getParameters().putParcelable(ImagePreviewFragment.PRESERVED_BITMAP, preservedBitmap);
            }

            if (dragInfo.getDropPositionX() < viewPager.getWidth() / 2f) {
                editorHelper.addPageToTheLeft(page, null);
            } else {
                editorHelper.addPageToTheRight(page, null);
            }

            dropped = true;
        } finally {
            finalizeRearrangement();
        }
    }

    private void finalizeRearrangement() {
        dragInfo = null;
    }

    /**
     * @return {@code true} if drag is currently active, {@code false} otherwise
     */
    public boolean isDragging() {
        return dragActive;
    }

    /**
     * Changes current item of {@link net.doo.snap.lib.ui.widget.ViewPager}
     */
    private class ScrollRunnable implements Runnable {

        private final int pageModifier;

        /**
         * @param pageModifier amount of pages to scroll (negative value will scroll pages to the left)
         */
        ScrollRunnable(int pageModifier) {
            this.pageModifier = pageModifier;
        }

        @Override
        public void run() {
            final int newItem = viewPager.getCurrentItem() + pageModifier;
            if (newItem >= 0 && newItem < adapter.getCount() - 1) {
                viewPager.setCurrentItem(newItem, true);
                pagesScrollHandler.postDelayed(this, SCROLL_DELAY_MILLIS);
            }
        }
    }

    /**
     * Contains information about performed drag
     */
    private class DragInfo {

        private final DragLocalState dragLocalState;
        private float x;
        private float y;

        private DragInfo(DragLocalState dragLocalState) {
            this.dragLocalState = dragLocalState;
        }

        /**
         * @return {@link net.doo.snap.lib.snap.edit.drag.DragLocalState} of performed drag
         */
        public DragLocalState getDragLocalState() {
            return dragLocalState;
        }

        /**
         * @return X position of drop event
         */
        public float getDropPositionX() {
            return x;
        }

        /**
         * @return Y position of drop event
         */
        public float getDropPositionY() {
            return y;
        }

        /**
         * @param x X position of drop event
         * @param y Y position of drop event
         */
        public void setDropPosition(float x, float y) {
            this.x = x;
            this.y = y;
        }

    }

    /**
     * Shadow builder for drag-n-drop framework
     */
    private class PageShadowBuilder extends View.DragShadowBuilder {

        public PageShadowBuilder(View view) {
            super(view);
        }

        @Override
        public void onDrawShadow(Canvas canvas) {
            final View view = getView();
            if (view == null) {
                super.onDrawShadow(canvas);
                return;
            }

            canvas.save();
            canvas.scale(
                    DRAG_SHADOW_SCALE, DRAG_SHADOW_SCALE,
                    view.getWidth() / 2f, view.getHeight() / 2f
            );
            super.onDrawShadow(canvas);
            canvas.restore();
        }
    }

}
