package net.doo.snap.lib.persistence;

import android.util.SparseArray;

import net.doo.snap.lib.detector.ContourDetector;

/**
 * {@link net.doo.snap.lib.persistence.Page} optimization types supported by native part
 */
public enum OptimizationType {
    NONE(ContourDetector.IMAGE_FILTER_NONE, "none"),
    COLOR_ENHANCED(ContourDetector.IMAGE_FILTER_COLOR_ENHANCED, "color_enhanced"),
    GRAYSCALE(ContourDetector.IMAGE_FILTER_GRAY, "grayscale"),
    BLACK_AND_WHITE(ContourDetector.IMAGE_FILTER_BINARIZED, "black_and_white");

    private static final SparseArray<OptimizationType> code2Type = new SparseArray<OptimizationType>();

    static {
        for (OptimizationType type : OptimizationType.values()) {
            code2Type.put(type.getCode(), type);
        }
    }

    private final int code;
    private final String name;

    OptimizationType(int code, String name) {
        this.code = code;
        this.name = name;
    }

    /**
     * @return integer parameter that should be passed to {@link net.doo.snap.lib.detector.ContourDetector} and used in persistence
     */
    public int getCode() {
        return code;
    }

    /**
     * @return name of {@link net.doo.snap.lib.persistence.OptimizationType} which doesn't depends on instance name
     */
    public String getName() {
        return name;
    }

    /**
     * @param code
     * @return {@link net.doo.snap.lib.persistence.OptimizationType} corresponding to the given code
     * @throws java.lang.IllegalStateException if no {@link net.doo.snap.lib.persistence.OptimizationType} was found for a given code
     */
    public static OptimizationType getByCode(int code) {
        OptimizationType type = code2Type.get(code);
        if (type == null) {
            throw new IllegalStateException("No Optimization type for code: " + code);
        }

        return type;
    }
}
