package net.doo.snap.lib.util;

import android.content.Context;
import android.os.Environment;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.widget.Toast;

import net.doo.snap.lib.R;

import java.io.File;
import java.io.IOException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Contains helper methods for working with files and directories
 */
public class FileUtils {
    private static final String FILE_NAME_EXCEPTIONS_REGULAR_EXPRESSION = "[\\\\|?*<\\\">+\\[\\]/':]";
    private static final Pattern FILE_NAME_ANTIPATTERN = Pattern.compile(FILE_NAME_EXCEPTIONS_REGULAR_EXPRESSION, Pattern.CASE_INSENSITIVE);
    private static final int MAX_FILE_NAME_LENGTH = 250;

    /**
     * @return {@code true} if filename can be used on filesystem. {@code false} otherwise.
     */
    public static boolean isFileNameSafe(CharSequence name) {
        /**
         * To avoid {@link java.lang.NullPointerException} in matcher
         */
        if (TextUtils.isEmpty(name) || name.length() > MAX_FILE_NAME_LENGTH) {
            return false;
        }

        Matcher matcher = FILE_NAME_ANTIPATTERN.matcher(name);
        return !matcher.find();
    }

    /**
     * Returns external directory with provided name (creates if needed). On failure shows toast message.
     *
     * @param context
     * @param directoryName name of the directory being created
     * @return external directory with provided name (creates if needed)
     * @throws IOException in case if IO failure
     */
    public static File getExternalFilesDirOrShowError(final Context context, String directoryName) throws IOException {
        File externalFilesDir = context.getExternalFilesDir(null);
        if (externalFilesDir == null) {
            return failAndShowErrorMessage(context);
        }

        File result = new File(externalFilesDir, directoryName);
        if (!result.exists() && !result.mkdir()) {
            return failAndShowErrorMessage(context);
        }

        return result;
    }

    /**
     * Returns external directory with provided name (creates if needed). On failure shows toast message.
     *
     * @param context
     * @param directoryName name of the directory being created
     * @return external directory with provided name (creates if needed)
     * @throws IOException in case if IO failure
     */
    public static File getExternalStorageDirOrShowError(final Context context, String directoryName) throws IOException {
        File externalFilesDir = Environment.getExternalStorageDirectory();
        if (externalFilesDir == null) {
            return failAndShowErrorMessage(context);
        }

        File result = new File(externalFilesDir, directoryName);
        if (!result.exists() && !result.mkdir()) {
            return failAndShowErrorMessage(context);
        }

        return result;
    }

    /**
     * Returns external cache directory with provided name (creates if needed). On failure shows toast message.
     *
     * @param context
     * @param directoryName name of the directory being created
     * @return external directory with provided name (creates if needed)
     * @throws IOException in case if IO failure
     */
    public static File getExternalCacheDirOrShowError(final Context context, String directoryName) throws IOException {
        File externalCacheDir = context.getExternalCacheDir();
        if (externalCacheDir == null) {
            return failAndShowErrorMessage(context);
        }

        File result = new File(externalCacheDir, directoryName);
        if (!result.exists() && !result.mkdir()) {
            return failAndShowErrorMessage(context);
        }

        return result;
    }

    private static File failAndShowErrorMessage(final Context context) throws IOException {
        new Handler(Looper.getMainLooper()).post(new Runnable() {
            @Override
            public void run() {
                Toast.makeText(context, context.getString(R.string.file_creation_error_message), Toast.LENGTH_LONG).show();
            }
        });

        throw new IOException("can't get external directory");
    }

}
