package net.doo.snap.lib.ui.widget.signature;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.PorterDuff;
import android.graphics.RectF;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;

import net.doo.snap.lib.R;
import net.doo.snap.lib.util.bitmap.BitmapUtils;

/**
 * Lets user draw a signature and export it as a {@link android.graphics.Bitmap}
 */
public class SignatureView extends View {

    private Paint paint = new Paint();
    private Path path = new Path();

    /**
     * Optimizes painting by invalidating the smallest possible area.
     */
    private float lastTouchX;
    private float lastTouchY;

    private Bitmap bitmap;
    private Canvas bitmapCanvas;

    private boolean hasContent = false;

    public SignatureView(Context context) {
        super(context);

        initPaint();
    }

    public SignatureView(Context context, AttributeSet attrs) {
        super(context, attrs);

        initPaint();

        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.SignatureView);
        try {
            paint.setStrokeWidth(
                    a.getDimension(R.styleable.SignatureView_signatureWidth, 0f)
            );
            paint.setColor(
                    a.getColor(R.styleable.SignatureView_signatureColor, Color.BLACK)
            );
        } finally {
            a.recycle();
        }
    }

    private void initPaint() {
        paint.setAntiAlias(true);
        paint.setStyle(Paint.Style.STROKE);
        paint.setStrokeJoin(Paint.Join.ROUND);
    }

    /**
     * Erases the signature.
     */
    public void clear() {
        path.reset();

        if (bitmap != null) {
            bitmapCanvas.drawColor(Color.BLACK, PorterDuff.Mode.CLEAR);
        }

        hasContent = false;

        invalidate();
    }

    /**
     * Assigns {@link android.graphics.Bitmap} which will be used as signature and might be updated.
     * If parameter is {@code null} clears current signature.
     */
    public void setSignature(Bitmap input) {
        if (!BitmapUtils.isBitmapValid(input)) {
            clear();
            return;
        }

        this.bitmap = input.copy(Bitmap.Config.ARGB_8888, true);

        this.bitmapCanvas = new Canvas(this.bitmap);
        hasContent = true;

        invalidate();
    }

    /**
     * @return {@link android.graphics.Bitmap} with drawn signature or {@code null} if nothing is drawn yet.
     */
    public Bitmap getSignature() {
        if (!hasContent || !BitmapUtils.isBitmapValid(bitmap)) {
            return null;
        }

        return bitmap;
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);

        bitmap = Bitmap.createBitmap(w, h, Bitmap.Config.ARGB_8888);
        bitmapCanvas = new Canvas(bitmap);
        hasContent = false;
    }

    @Override
    protected void onDraw(Canvas canvas) {
        canvas.drawBitmap(bitmap, 0f, 0f, null);
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        float eventX = event.getX();
        float eventY = event.getY();

        switch (event.getAction()) {
            case MotionEvent.ACTION_DOWN:
                path.moveTo(eventX, eventY);
                lastTouchX = eventX;
                lastTouchY = eventY;
                // There is no end point yet, so don't waste cycles invalidating.
                return true;

            case MotionEvent.ACTION_MOVE:
            case MotionEvent.ACTION_UP:
                final float pointX = (eventX + lastTouchX) / 2f;
                final float pointY = (eventY + lastTouchY) / 2f;

                path.quadTo(
                        lastTouchX, lastTouchY,
                        pointX, pointY
                );

                if (bitmap != null) {
                    bitmapCanvas.drawPath(path, paint);
                }

                path.reset();
                path.moveTo(pointX, pointY);

                lastTouchX = eventX;
                lastTouchY = eventY;

                hasContent = true;

                invalidate();

                return true;

            default:
                return false;
        }
    }

}