package net.doo.snap.lib.snap.camera;

import android.animation.ObjectAnimator;
import android.animation.TypeEvaluator;
import android.animation.ValueAnimator;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.Rect;
import android.hardware.Camera;
import android.os.Handler;
import android.util.Property;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.View;
import android.widget.Toast;

import com.commonsware.cwac.camera.CameraView;

import net.doo.snap.lib.Constants;
import net.doo.snap.lib.R;
import net.doo.snap.lib.detector.CameraDetectorListener;
import net.doo.snap.lib.detector.DetectionResult;
import net.doo.snap.lib.sensor.SignificantMoveListener;
import net.doo.snap.lib.snap.AutoSnappingDetectionHelper;
import net.doo.snap.lib.util.CameraConfiguration;
import net.doo.snap.lib.util.ThemesHelper;
import net.doo.snap.lib.util.log.DebugLog;
import net.doo.snap.lib.util.snap.PolygonHelper;
import net.doo.snap.lib.util.snap.Utils;

import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Implementation of {@link com.commonsware.cwac.camera.CameraView}
 * Draws detected image analysis results
 *
 * @see net.doo.snap.lib.detector.ContourDetector
 * @see DetectionTask
 * @see CameraPreviewFragment
 */
public class SnapCameraView extends CameraView implements CameraDetectorListener, SignificantMoveListener {

    private static final Property<Paint, Integer> PAINT_ALPHA_PROPERTY = Property.of(Paint.class, Integer.TYPE, "alpha");
    private static final int POLYGON_FADE_OUT_DURATION_MS = 250;
    private static final int DELAY_AFTER_MOVE_MS = 2000;

    private static final int DELAY_AUTO_SNAPPING_SHOOT_MS = 1000;

    private static final int MIN_AREA_COORD = -1000;
    private static final int MAX_AREA_COORD = 1000;

    private static final int AREA_WEIGHT = 1000;
    private static final int DEFAULT_FOCUS_AREA_SIZE = 75;
    private static final int HIDE_TOUCH_FOCUS_DELAY_MS = 1000;
    private static final int DELAY_REFOCUSING_NEEDED_MS = 3000;
    private static final int AUTO_FOCUS_TIMEOUT_MS = 5000;

    private static final String SAMSUNG_SCENE_MODE_TEXT = "text";

    private List<PointF> polygon;
    private final Paint paint;
    private boolean isAutoSnapEnabled = true;
    private boolean isAutoFocusing = false;

    private ValueAnimator polygonAnimator;
    private ObjectAnimator polygonAlphaAnimator;
    private SnapCameraView.PolygonEvaluator polygonEvaluator = new PolygonEvaluator();

    private AutoSnappingDetectionHelper detectionHelper;
    private CameraConfiguration cameraConfiguration;

    private boolean cameraAvailable = false;

    private long lastSignificantMove;

    /**
     * points to be drawn on canvas for better performance
     */
    private float[] points;
    private float[] animationPoints;

    private PolygonHelper polygonHelper;
    private boolean drawPolygon = true;

    private byte[] previewBuffer;

    private boolean isAutosnappingShooting = false;
    private long lastFocusedTime;

    private Rect touchRect;
    private final Paint touchFocusPaint;

    private Handler autoFocusHandler = new Handler();
    private Runnable cancelAutoFocusRunnable = new Runnable() {
        @Override
        public void run() {
            cancelAutoFocus();
        }
    };

    private OnReadyToSnapListener onReadyToSnapListener;

    private final Runnable autosnappingShoot = new Runnable() {
        @Override
        public void run() {
            if (onReadyToSnapListener != null) {
                onReadyToSnapListener.readyToSnap(false);
            }

            boolean movedRecently = System.currentTimeMillis() - lastSignificantMove <= DELAY_AFTER_MOVE_MS;
            SnapCameraHost host = (SnapCameraHost) getHost();
            if (!movedRecently
                    && isAutoSnapEnabled
                    && host.useContourDetection()
                    && isAutoFocusAvailable()) {

                if (System.currentTimeMillis() - lastFocusedTime > DELAY_REFOCUSING_NEEDED_MS) {
                    autoFocus();
                } else {
                    ((SnapCameraHost) getHost()).takePicture();
                    isAutosnappingShooting = false;
                }

            } else {
                isAutosnappingShooting = false;
            }
        }
    };

    public SnapCameraView(Context context, AutoSnappingDetectionHelper detectionHelper, CameraConfiguration cameraConfiguration) {
        super(context);

        this.detectionHelper = detectionHelper;
        this.cameraConfiguration = cameraConfiguration;

        paint = new Paint();
        paint.setColor(getResources().getColor(ThemesHelper.getResourceId(context, R.attr.accent_color)));
        paint.setStyle(Paint.Style.STROKE);
        paint.setStrokeWidth(getResources().getDimension(R.dimen.polygon_stroke_width));
        paint.setAntiAlias(true);

        touchFocusPaint = new Paint();
        touchFocusPaint.setColor(getResources().getColor(android.R.color.white));
        touchFocusPaint.setStyle(Paint.Style.STROKE);
        touchFocusPaint.setStrokeWidth(getResources().getDimension(R.dimen.touch_focus_polygon_width));
        touchFocusPaint.setAntiAlias(true);

        points = new float[16];
        animationPoints = new float[points.length];

        polygonHelper = new PolygonHelper();
        polygon = Collections.emptyList();
    }

    /**
     * @return polygon containing document inside snapped image
     */
    public List<PointF> getPolygon() {
        return polygon;
    }

    /**
     * Sets option to snap image as soon as good polygon is detected
     *
     * @param enabled
     */
    public void setAutoSnapEnabled(boolean enabled) {
        isAutoSnapEnabled = enabled;
        if (!isAutoSnapEnabled) {
            detectionHelper.reset(true);
        }
    }

    @Override
    protected void dispatchDraw(Canvas canvas) {
        super.dispatchDraw(canvas);

        if (drawPolygon && !polygon.isEmpty()) {
            canvas.drawLines(points, paint);
        }
        if (touchRect != null) {
            canvas.drawRect(touchRect, touchFocusPaint);
        }
    }

    @Override
    public void onCameraOpen(Camera camera) throws RuntimeException {
        super.onCameraOpen(camera);
        if(camera != null) {
            Camera.Parameters parameters = camera.getParameters();
            if (parameters != null && !cameraConfiguration.isConigured()) {
                cameraConfiguration.loadCameraParameters(parameters);
            }
        }
    }

    @Override
    protected void onLayout(boolean changed, int l, int t, int r, int b) {
        super.onLayout(changed, l, t, r, b);
        View view = getChildAt(0);
        if (view != null) {
            polygonHelper.setLayout(view.getLeft(), view.getTop(), view.getRight(), view.getBottom());
        }
    }

    @Override
    public void autoFocus() {
        autoFocusHandler.removeCallbacks(cancelAutoFocusRunnable);
        autoFocusHandler.postDelayed(cancelAutoFocusRunnable, AUTO_FOCUS_TIMEOUT_MS);

        isAutoFocusing = true;

        try {
            Camera.Parameters params = getCameraParameters();
            Utils.enableAutoFocus(params);
            setFocusAndMeteringArea(params);
            setCameraParameters(params);
            super.autoFocus();
        } catch (RuntimeException e) {
            DebugLog.logException(e);
        }
    }

    @Override
    public void cancelAutoFocus() {
        super.cancelAutoFocus();
        updateTouchRect(false);
        isAutoFocusing = false;
        autoFocusHandler.removeCallbacks(cancelAutoFocusRunnable);
    }

    @Override
    public boolean isAutoFocusAvailable() {
        return super.isAutoFocusAvailable() && !isAutoFocusing;
    }

    /**
     * Sets area to focus. Based on current polygon
     * @param parameters
     */
    private void setFocusAndMeteringArea(Camera.Parameters parameters) {
        List<Camera.Area> areas = null;

        if (parameters.getMaxNumFocusAreas() > 0) {
            areas = getAreas();
            parameters.setFocusAreas(areas);
        }

        if (parameters.getMaxNumMeteringAreas() > 0) {
            areas = getAreas();
            parameters.setMeteringAreas(areas);
        }
    }

    private List<Camera.Area> getAreas() {
        Rect areaRect;
        List<Camera.Area> areas;
        if (touchRect != null) {
            areaRect = new Rect(
                    polygonCoordToAreaCoord(touchRect.left / (float) getWidth()),
                    polygonCoordToAreaCoord(touchRect.top / (float) getHeight()),
                    polygonCoordToAreaCoord(touchRect.right / (float) getWidth()),
                    polygonCoordToAreaCoord(touchRect.bottom / (float) getHeight()));
        } else if (!polygon.isEmpty()) {
            float minX = 1;
            float maxX = 0;
            float minY = 1;
            float maxY = 0;

            for (PointF point : polygon) {
                if (point.x < minX) minX = point.x;
                if (point.x > maxX) maxX = point.x;
                if (point.y < minY) minY = point.y;
                if (point.y > maxY) maxY = point.y;
            }

            float left = (maxX - minX) / 2 - 0.001f * DEFAULT_FOCUS_AREA_SIZE;
            float right = (maxX - minX) / 2 + 0.001f * DEFAULT_FOCUS_AREA_SIZE;
            float top = minY;
            float bottom = maxY;

            areaRect = new Rect(
                    polygonCoordToAreaCoord(left),
                    polygonCoordToAreaCoord(top),
                    polygonCoordToAreaCoord(right),
                    polygonCoordToAreaCoord(bottom));

        } else {
            areaRect = new Rect(
                    (-1) * DEFAULT_FOCUS_AREA_SIZE,
                    (-1) * DEFAULT_FOCUS_AREA_SIZE,
                    DEFAULT_FOCUS_AREA_SIZE,
                    DEFAULT_FOCUS_AREA_SIZE);

        }

        // weight can be from 1 to 1000, let's set maximum
        Camera.Area area = new Camera.Area(areaRect, AREA_WEIGHT);
        areas = new ArrayList<>();
        areas.add(area);
        return areas;
    }

    /**
     * Transforms 0..1 float coordinate to integer -1000..1000 coordinate
     * @param coordinate
     * @return
     */
    private int polygonCoordToAreaCoord(float coordinate) {
        int result = Math.round(coordinate * MAX_AREA_COORD * 2 + MIN_AREA_COORD);
        if (result > MAX_AREA_COORD) result = MAX_AREA_COORD;
        if (result < MIN_AREA_COORD) result = MIN_AREA_COORD;
        return result;
    }

    /**
     * Starts auto focusing on touch area.
     * @param event touch motion event
     * @return
     */
    public void autoFocusOnTouch(MotionEvent event) {
        touchFocusPaint.setColor(getResources().getColor(android.R.color.white));

        float x = event.getX();
        float y = event.getY();

        touchRect = new Rect(
                (int) (x - DEFAULT_FOCUS_AREA_SIZE),
                (int) (y - DEFAULT_FOCUS_AREA_SIZE),
                (int) (x + DEFAULT_FOCUS_AREA_SIZE),
                (int) (y + DEFAULT_FOCUS_AREA_SIZE));

        invalidate();
        autoFocus();
    }

    @Override
    public void onAutoFocus(boolean success, Camera camera) {
        /**
         * Using isAutoFocusing flag to ensure we called {@link #autoFocus()}
         * before getting this callback
         */
        if (isAutoFocusing) {
            updateTouchRect(success);
            autoFocusHandler.removeCallbacks(cancelAutoFocusRunnable);

            isAutoFocusing = false;
            super.onAutoFocus(success, camera);
        }

        lastFocusedTime = success ? System.currentTimeMillis() : 0;

        if (isAutosnappingShooting) {
            if (isAutoFocusAvailable()
                    && success) {
                ((SnapCameraHost) getHost()).takePicture();
            } else {
                isAutosnappingShooting = false;
            }
        }
    }

    private void updateTouchRect(boolean success) {
        if (touchRect != null) {
            if (touchFocusPaint.getColor() != Color.WHITE) {
                touchFocusPaint.setAlpha(0);
                invalidate();
                return;
            }

            touchFocusPaint.setColor(getResources().getColor(
                    success
                            ? android.R.color.holo_green_light
                            : android.R.color.holo_red_light));
            invalidate();
            postDelayed(new Runnable() {
                @Override
                public void run() {
                    if (touchFocusPaint.getColor() != Color.WHITE) {
                        touchFocusPaint.setAlpha(0);
                        invalidate();
                    }
                }
            }, HIDE_TOUCH_FOCUS_DELAY_MS);
        }
    }

    @Override
    public void onDetectionOK(final List<PointF> polygon) {
        if (!this.polygon.isEmpty()) {
            animateToPolygon(polygon);
        } else {
            polygonHelper.polygonToPoints(polygon, points);
        }

        this.polygon = polygon;

        detectionHelper.onResult(DetectionResult.OK);
        if (!isAutosnappingShooting) {
            isAutosnappingShooting = true;
            Toast toast = Toast.makeText(getContext(), R.string.autosnapping_hint_do_not_move, Toast.LENGTH_SHORT);
            toast.setGravity(Gravity.CENTER, 0, 0);
            toast.show();
            if (onReadyToSnapListener != null) {
                onReadyToSnapListener.readyToSnap(true);
            }
            postDelayed(autosnappingShoot, DELAY_AUTO_SNAPPING_SHOOT_MS);
        }

        addPreviewCallbackBuffer(previewBuffer);
    }

    @Override
    public void onBarcodeDetectionOK(String content) {
        //continue
    }

    @Override
    public void onDetectionWithError(DetectionResult result, List<PointF> polygon) {
        if (isAutoSnapEnabled) {
            detectionHelper.onResult(result);
        }

        this.polygon = polygon;
        animateToPolygon(polygon);

        addPreviewCallbackBuffer(previewBuffer);
    }

    @Override
    public void onDetectionFailed(DetectionResult result) {
        if (isAutoSnapEnabled) {
            detectionHelper.onResult(result);
        }

        animateToPolygon(Collections.<PointF>emptyList());

        addPreviewCallbackBuffer(previewBuffer);
    }

    private void animateToPolygon(List<PointF> polygon) {
        if (polygonAnimator != null) {
            polygonAnimator.cancel();
            polygonAnimator = null;
        }

        if (polygonAlphaAnimator != null) {
            polygonAlphaAnimator.cancel();
            polygonAlphaAnimator = null;
        }

        if (polygon.isEmpty()) {
            animatePolygonFadeOut();
            return;
        }

        paint.setAlpha(255);

        polygonHelper.polygonToPoints(polygon, animationPoints);

        polygonAnimator = ValueAnimator.ofObject(
                polygonEvaluator,
                points, animationPoints
        );
        polygonAnimator.start();
    }

    private void animatePolygonFadeOut() {
        polygonAlphaAnimator = ObjectAnimator.ofInt(paint, PAINT_ALPHA_PROPERTY, 0);

        polygonAlphaAnimator.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
            @Override
            public void onAnimationUpdate(ValueAnimator valueAnimator) {
                invalidate();
            }
        });

        polygonAlphaAnimator.setDuration(POLYGON_FADE_OUT_DURATION_MS);

        polygonAlphaAnimator.start();
    }

    /**
     * Hides all notifications that help while auto snapping
     * Dismisses toast, hides "Move closer" arrows and "Rotate" image
     */
    public void resetNotifications() {
        detectionHelper.reset(true);
    }

    @Override
    public void onResume() {
        super.onResume();
        cameraAvailable = true;
    }

    @Override
    public void onPause() {
        super.onPause();
        cameraAvailable = false;
    }

    @Override
    public void startPreview() {
        if (cameraAvailable) {
            polygon = Collections.emptyList();
            touchRect = null;
            isAutoFocusing = false;
            Camera.Parameters parameters = getCameraParameters();
            if (parameters != null) {
                super.startPreview();
                setAdvancedParameter(parameters);
                setFocusAndMeteringArea(parameters);
                setCameraParameters(parameters);

                previewBuffer = ((SnapCameraHost) getHost()).getNewPreviewBuffer();
                addPreviewCallbackBuffer(previewBuffer);

                polygonHelper.setRotation(getDisplayOrientation());

                isAutosnappingShooting = false;
            }
        }
    }

    private void setAdvancedParameter(@NotNull Camera.Parameters parameters) {
        parameters.setJpegQuality(Constants.MAX_JPEG_QUALITY);
        List<String> sceneModes = parameters.getSupportedSceneModes();
        if (sceneModes != null) {
            if (sceneModes.contains(SAMSUNG_SCENE_MODE_TEXT)) {
                parameters.setSceneMode(SAMSUNG_SCENE_MODE_TEXT);
            }
        }
    }

    @Override
    public void stopPreview() {
        if (cameraAvailable) {
            super.stopPreview();
        }
    }

    /**
     * Indicates whether polygon should be drawn or not.
     *
     * @param drawPolygon
     */
    public void setDrawPolygon(boolean drawPolygon) {
        if (drawPolygon && !this.drawPolygon) {
            addPreviewCallbackBuffer(previewBuffer);
        }
        this.drawPolygon = drawPolygon;
        invalidate();
    }

    @Override
    public float getSignificantMoveThreshold() {
        return SignificantMoveListener.THRESHOLD_LOW;
    }

    @Override
    public void onSignificantMove() {
        lastSignificantMove = System.currentTimeMillis();
    }

    /**
     * Sets listener to notify about ready to snap status change
     * @param listener
     */
    public void setOnReadyToSnapListener(OnReadyToSnapListener listener) {
        onReadyToSnapListener = listener;
    }

    /**
     * Performs linear translation of arbitrary float array values to polygon drawing points
     */
    private class PolygonEvaluator implements TypeEvaluator {

        @Override
        public Object evaluate(float fraction, Object startObj, Object endObj) {
            evaluatePolygon(
                    fraction,
                    (float[]) startObj,
                    (float[]) endObj
            );

            return null;
        }

        private void evaluatePolygon(float fraction, float[] start, float[] end) {
            for (int i = 0; i < points.length; i++) {
                points[i] = start[i] + fraction * (end[i] - start[i]);
            }

            invalidate();
        }
    }

    /**
     * Used to notify when camera is about to shoot
     */
    public interface OnReadyToSnapListener {
        /**
         *
         * @param ready is {@code true} when camera is ready to shoot in autosnapping mode, and {@code false} if camera is not shooting in autosnapping mode
         */
        void readyToSnap(boolean ready);
    }
}
