package net.doo.snap.lib.util.billing;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Represents a block of information about in-app items.
 * An Inventory is returned by such methods as {@link IabHelper#queryInventory}.
 */
public class Inventory {
    private final Map<String, SkuDetails> skuMap = new HashMap<String, SkuDetails>();
    private final Map<String, Purchase> purchaseMap = new HashMap<String, Purchase>();

    Inventory() {
    }

    /**
     * Returns the listing details for an in-app product.
     */
    public SkuDetails getSkuDetails(String sku) {
        return skuMap.get(sku);
    }

    /**
     * Returns purchase information for a given product, or null if there is no purchase.
     */
    public Purchase getPurchase(String sku) {
        return purchaseMap.get(sku);
    }

    /**
     * Returns whether or not there exists a purchase of the given product.
     */
    public boolean hasPurchase(String sku) {
        return purchaseMap.containsKey(sku);
    }

    /**
     * Return whether or not details about the given product are available.
     */
    public boolean hasDetails(String sku) {
        return skuMap.containsKey(sku);
    }

    /**
     * Erase a purchase (locally) from the inventory, given its product ID. This just
     * modifies the Inventory object locally and has no effect on the server! This is
     * useful when you have an existing Inventory object which you know to be up to date,
     * and you have just consumed an item successfully, which means that erasing its
     * purchase data from the Inventory you already have is quicker than querying for
     * a new Inventory.
     */
    public void erasePurchase(String sku) {
        purchaseMap.remove(sku);
    }

    /**
     * Returns a list of all owned product IDs.
     */
    public List<String> getAllOwnedSkus() {
        return new ArrayList<String>(purchaseMap.keySet());
    }

    /**
     * Returns a list of all owned product IDs of a given type
     */
    public List<String> getAllOwnedSkus(String itemType) {
        List<String> result = new ArrayList<String>();
        for (Purchase purchase : purchaseMap.values()) {
            if (purchase.getItemType().equals(itemType)) {
                result.add(purchase.getSku());
            }
        }
        return result;
    }

    /**
     * Returns a list of all purchases.
     */
    public List<Purchase> getAllPurchases() {
        return new ArrayList<Purchase>(purchaseMap.values());
    }

    /**
     * Adds sku details
     *
     * @param skuDetails
     */
    public void addSkuDetails(SkuDetails skuDetails) {
        skuMap.put(skuDetails.getSku(), skuDetails);
    }

    /**
     * Adds purchase
     *
     * @param purchase
     */
    public void addPurchase(Purchase purchase) {
        purchaseMap.put(purchase.getSku(), purchase);
    }
}
