package net.doo.snap.lib.ui.widget;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.drawable.Drawable;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.widget.ImageView;

/**
 * Let user place and scale arbitrary block (represented as {@link android.graphics.drawable.Drawable})
 * on image.
 * <p/>
 * Provides result through following methods:
 * {@link #getBlockOrigin()}
 * {@link #getBlockScale()}
 */
public class AlignBlockImageView extends ImageView {

    private Drawable block;
    private Drawable background;

    private float historicalX;
    private float historicalY;
    private Rect blockOriginalBounds = new Rect();
    private Rect blockBounds = new Rect();
    private Rect viewBounds = new Rect();
    private Rect backgroundPadding = new Rect();

    private Matrix blockMatrix = new Matrix();
    private float blockScale = 1f;

    private Rect tempRect = new Rect();
    private RectF tempRectF = new RectF();

    private Runnable onSizeChangedRunnable;

    public AlignBlockImageView(Context context) {
        super(context);
    }

    public AlignBlockImageView(Context context, AttributeSet attrs) {
        super(context, attrs);
    }

    public AlignBlockImageView(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
    }

    /**
     * Adds {@link android.graphics.drawable.Drawable} as moving block to this {@link net.doo.snap.lib.ui.widget.AlignBlockImageView}
     * which will be placed in the center of this view with default scale.
     */
    public void setBlock(Drawable drawable) {
        block = drawable;

        if (block == null) {
            invalidate();
            return;
        }

        onSizeChangedRunnable = new Runnable() {
            @Override
            public void run() {
                int left = (getWidth() - block.getIntrinsicWidth()) / 2;
                int top = (getHeight() - block.getIntrinsicHeight()) / 2;

                alignBlock(left, top, 1f);
            }
        };

        if (getWidth() > 0 && getHeight() > 0) {
            onSizeChangedRunnable.run();
            onSizeChangedRunnable = null;
        }
    }

    /**
     * Adds {@link android.graphics.drawable.Drawable} as moving block to this {@link net.doo.snap.lib.ui.widget.AlignBlockImageView}.
     *
     * @param drawable moving block
     * @param origin   origin to which block will be translated.
     * @param scale    scale of the block
     */
    public void setBlock(Drawable drawable, final PointF origin, final float scale) {
        block = drawable;

        if (block == null) {
            invalidate();
            return;
        }

        onSizeChangedRunnable = new Runnable() {
            @Override
            public void run() {
                alignBlock(
                        (int) (origin.x * getWidth()),
                        (int) (origin.y * getHeight()),
                        scale
                );
            }
        };

        if (getWidth() > 0 && getHeight() > 0) {
            onSizeChangedRunnable.run();
            onSizeChangedRunnable = null;
        }
    }

    private void alignBlock(int left, int top, float scale) {
        if (block != null) {
            blockOriginalBounds.set(
                    left, top,
                    left + block.getIntrinsicWidth(), top + block.getIntrinsicHeight()
            );
            blockBounds.set(blockOriginalBounds);

            block.setBounds(blockBounds);

            setBlockScale(scale);

            alignBackground();
        }

        invalidate();
    }

    /**
     * Provides origin point for current point.
     * Values for coordinates are in range 0..1. Value represents fraction of {@link net.doo.snap.lib.ui.widget.AlignBlockImageView}
     * dimension.
     *
     * @return {@link android.graphics.PointF} origin of this block. Default value is PointF(0, 0).
     */
    public PointF getBlockOrigin() {
        PointF origin = new PointF(0f, 0f);
        if (block == null) {
            return origin;
        }

        origin.set(
                blockBounds.left / (float) getWidth(),
                blockBounds.top / (float) getHeight()
        );

        return origin;
    }

    /**
     * @return scale of the block. Default value is 1f.
     */
    public float getBlockScale() {
        return block != null
                ? blockScale
                : 1f;
    }

    /**
     * Assigns block scale.
     *
     * @return {@code true} if block was scaled. {@code false} if scaled block didn't fit into view bounds
     * and therefore won't be scaled.
     */
    public boolean setBlockScale(float scale) {
        if (viewBounds.isEmpty() || block == null) {
            return false;
        }

        this.blockScale = scale;

        applyScale(scale);

        if (!viewBounds.contains(tempRect)) {
            tempRect.offset(
                    (int) Math.min(0f, viewBounds.right - tempRect.right - 1),
                    (int) Math.min(0f, viewBounds.bottom - tempRect.bottom - 1)
            );

            if (!viewBounds.contains(tempRect)) {
                return false;
            }
        }

        blockBounds.set(tempRect);
        block.setBounds(blockBounds);
        alignBackground();

        invalidate();

        return true;
    }

    private void applyScale(float scale) {
        tempRectF.set(blockOriginalBounds);

        blockMatrix.setScale(scale, scale);
        blockMatrix.mapRect(tempRectF);

        tempRect.set(
                (int) tempRectF.left,
                (int) tempRectF.top,
                (int) tempRectF.right,
                (int) tempRectF.bottom
        );
        tempRect.offsetTo(
                blockBounds.left,
                blockBounds.top
        );
    }

    @Override
    public void setBackgroundDrawable(Drawable background) {
        setBackground(background);
    }

    @Override
    public void setBackground(Drawable background) {
        this.background = background;
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);

        alignBackground();

        viewBounds.set(
                backgroundPadding.left, backgroundPadding.top,
                w - backgroundPadding.right, h - backgroundPadding.bottom
        );

        if (onSizeChangedRunnable != null) {
            onSizeChangedRunnable.run();
            onSizeChangedRunnable = null;
        }
    }

    @Override
    public void draw(Canvas canvas) {
        super.draw(canvas);

        if (block == null) {
            return;
        }

        block.draw(canvas);

        if (background != null) {
            background.draw(canvas);
        }
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        if (block == null) {
            return false;
        }

        switch (event.getAction()) {
            case MotionEvent.ACTION_DOWN:
                historicalX = event.getX();
                historicalY = event.getY();

                return blockBounds.contains(
                        (int) historicalX,
                        (int) historicalY
                );
            case MotionEvent.ACTION_MOVE:
                final int diffX = (int) (event.getX() - historicalX);
                final int diffY = (int) (event.getY() - historicalY);

                blockBounds.offset(diffX, 0);
                if (!viewBounds.contains(blockBounds)) {
                    blockBounds.offset(-diffX, 0);
                }

                blockBounds.offset(0, diffY);
                if (!viewBounds.contains(blockBounds)) {
                    blockBounds.offset(0, -diffY);
                }

                block.setBounds(blockBounds);

                alignBackground();

                invalidate();

                historicalX = event.getX();
                historicalY = event.getY();

                return true;
        }

        return false;
    }

    private void alignBackground() {
        if (background == null) {
            return;
        }

        background.getPadding(backgroundPadding);

        background.setBounds(
                blockBounds.left - backgroundPadding.left,
                blockBounds.top - backgroundPadding.top,
                blockBounds.right + backgroundPadding.right,
                blockBounds.bottom + backgroundPadding.bottom
        );
    }

}
