package net.doo.snap.lib.billing;

import android.util.Base64;

import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.util.UUID;

/**
 * Provides security methods for communication with billing services
 */
public final class SecurityManager {

    public static final String KEY_FACTORY_ALGORITHM = "RSA";
    public static final String SIGNATURE_ALGORITHM = "SHA1withRSA";

    //TODO obfuscate
    private static final String PUBLIC_KEY_BASE_64 = "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAu82nw1cLa7TrVGRF00wnHci9NqHuVAnH5EFeyoUKhLBX9duv6guVRktGx5ad7K35SPy8ZksaIvZnL6j2IbUtjweRPP+tXewONGhV4j+7wROQ2G4k2zbBVo7bKXzgKEfbpbKTbH4iub0xdBxsvW6X4OwbZ1zDrC+b4pHlddJpP167XouU7DCuaErLnnchf/ZA/2yha9s4boqIYty6e9hrOwjDf13Co05k+FWHMI5hQBCwL/4oljB/GXHs0mDvUsLa9vos3jjNxGht3aheZBlgLRyIht8u/hN1ZWs4RyUqI9Mmok3CSajbcnQj8PLzuY274iqklg4shgAzoVTOv3it4wIDAQAB";

    /**
     * @return application {@link java.security.PublicKey} for verification of signatures
     */
    public static PublicKey getPublicKey() throws NoSuchAlgorithmException, InvalidKeySpecException {
        byte[] key = Base64.decode(PUBLIC_KEY_BASE_64, Base64.DEFAULT);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_FACTORY_ALGORITHM);
        return keyFactory.generatePublic(new X509EncodedKeySpec(key));
    }

    /**
     * Verifies signature of message or throws exception on failure.
     *
     * @param message   message which signature has to be verified
     * @param signature signaure of provided message
     * @throws VerificationFailedException in case if signature verification fails for any reason
     */
    public void verifySignature(String message, String signature) throws VerificationFailedException {
        if (message == null || signature == null) {
            throw new VerificationFailedException();
        }

        Signature signatureAlgorithm = buildSignatureAlgorithm();

        verifySignature(
                signatureAlgorithm,
                message.getBytes(),
                Base64.decode(signature, Base64.DEFAULT)
        );
    }

    private void verifySignature(Signature signatureAlgorithm, byte[] messageBytes, byte[] signatureBytes) throws VerificationFailedException {
        try {
            signatureAlgorithm.update(messageBytes);

            if (!signatureAlgorithm.verify(signatureBytes)) {
                throw new VerificationFailedException();
            }
        } catch (SignatureException e) {
            throw new VerificationFailedException(e);
        }
    }

    private Signature buildSignatureAlgorithm() throws VerificationFailedException {
        try {
            Signature signatureAlgorithm = Signature.getInstance(SIGNATURE_ALGORITHM);
            signatureAlgorithm.initVerify(getPublicKey());

            return signatureAlgorithm;
        } catch (NoSuchAlgorithmException e) {
            throw new VerificationFailedException(e);
        } catch (InvalidKeyException e) {
            throw new VerificationFailedException(e);
        } catch (InvalidKeySpecException e) {
            throw new VerificationFailedException(e);
        }
    }

    /**
     * @return pseudo-unique {@link String} value
     */
    public String generateNonce() {
        return UUID.randomUUID().toString();
    }

    /**
     * Thrown when verification of message fails
     */
    public static class VerificationFailedException extends Exception {

        public VerificationFailedException(Throwable throwable) {
            super(throwable);
        }

        public VerificationFailedException() {
        }

    }

}
