package net.doo.snap.lib.util.bitmap;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Color;
import android.graphics.Rect;

import org.apache.commons.io.IOUtils;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

/**
 * Contains convenient methods for {@link android.graphics.Bitmap}s
 */
public class BitmapUtils {

    /**
     * @return bounds of {@link android.graphics.Bitmap} from file.
     */
    public static Rect getBitmapBounds(String path) {
        Rect bounds = new Rect();

        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;

        BitmapFactory.decodeFile(path, options);
        bounds.set(
                0, 0,
                options.outWidth, options.outHeight
        );

        return bounds;
    }

    /**
     * @return {@code true} if {@link android.graphics.Bitmap} valid (and can be drawed). {@code false}
     * otherwise.
     */
    public static boolean isBitmapValid(Bitmap bitmap) {
        return bitmap != null && !bitmap.isRecycled();
    }

    /**
     * @return {@link android.graphics.Rect} with minimum bounds in which all content of {@link android.graphics.Bitmap} might be fit.
     * Assuming that transparent pixels are "empty", therefore not counting them as a content. Method performs in linear time (relatively to
     * amount of pixels), so it's not recommended to invoke it from the main thread for large images.
     * @throws java.lang.NullPointerException if {@link android.graphics.Bitmap} is null
     */
    public static Rect getContentBounds(Bitmap bitmap) throws NullPointerException {
        if (bitmap == null) {
            throw new NullPointerException("Bitmap is null");
        }

        int minX = bitmap.getWidth();
        int minY = bitmap.getHeight();
        int maxX = 0;
        int maxY = 0;

        for (int y = 0; y < bitmap.getHeight(); y++) {
            for (int x = 0; x < bitmap.getWidth(); x++) {
                final int pixel = bitmap.getPixel(x, y);

                if (pixel != Color.TRANSPARENT) {
                    if (x < minX) minX = x;
                    if (y < minY) minY = y;
                    if (x > maxX) maxX = x;
                    if (y > maxY) maxY = y;
                }
            }
        }

        return new Rect(minX, minY, maxX, maxY);
    }

    /**
     * Compresses {@link android.graphics.Bitmap} to {@link java.io.File}
     */
    public static void compress(Bitmap bitmap, Bitmap.CompressFormat format, int quality, File file) throws IOException {
        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(file);
            bitmap.compress(format, quality, fos);
        } finally {
            IOUtils.closeQuietly(fos);
        }
    }

    public static Bitmap createPreview(byte[] image, int targetWidth, int targetHeight) {
        if (image == null) {
            return null;
        }
        final BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;
        BitmapFactory.decodeByteArray(image, 0, image.length, options);
        options.inSampleSize = calculateInSampleSize(options, targetWidth, targetHeight);
        options.inJustDecodeBounds = false;
        return BitmapFactory.decodeByteArray(image, 0, image.length, options);
    }

    private static int calculateInSampleSize(BitmapFactory.Options options, int reqWidth, int reqHeight) {
        int height = options.outHeight;
        int width = options.outWidth;
        boolean imageLandscape = width > height;
        boolean desiredLandscape = reqWidth > reqHeight;

        if (imageLandscape != desiredLandscape) {
            height = options.outWidth;
            width = options.outHeight;
        }

        int inSampleSize = 1;

        if (height > reqHeight || width > reqWidth) {

            int halfHeight = height / 2;
            int halfWidth = width / 2;

            while ((halfHeight / inSampleSize) > reqHeight
                    && (halfWidth / inSampleSize) > reqWidth) {
                inSampleSize *= 2;
            }
        }

        return inSampleSize;
    }

}
