package net.doo.snap.lib.snap.camera;

import android.graphics.PointF;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import com.google.zxing.*;
import com.google.zxing.common.HybridBinarizer;
import net.doo.snap.lib.detector.CameraDetectorListener;
import net.doo.snap.lib.detector.ContourDetector;
import net.doo.snap.lib.detector.DetectionResult;
import net.doo.snap.lib.snap.camera.barcode.BarcodeDetector;
import net.doo.snap.lib.util.log.DebugLog;

import java.util.*;

/**
 * Performs image analysis and notifies {@link net.doo.snap.lib.detector.CameraDetectorListener}
 *
 * @see SnapCameraHost
 * @see net.doo.snap.lib.detector.ContourDetector
 */
public class DetectionTask implements Runnable {
    private final BarcodeDetector barcodeDetector;

    private ContourDetector detector;
    private final byte[] image;
    private final int previewWidth;
    private final int previewHeight;
    private final CameraDetectorListener listener;
    private final boolean detectBarcode;

    /**
     *
     * @param barcodeDetector zxing muliformat barcode detector
     * @param detector native wrapper to detect documet contours
     * @param image byte array of image data in {@link android.graphics.ImageFormat#NV21} format
     * @param previewWidth width of preview obtained from {@link android.hardware.Camera.Parameters#getSupportedPreviewSizes()}
     * @param previewHeight height of preview obtained from {@link android.hardware.Camera.Parameters#getSupportedPreviewSizes()}
     * @param listener receives callback about detected polygon
     * @param detectBarcode {@code true} if barcode detection mode enabled, {@code false} - otherwise
     */
    public DetectionTask(BarcodeDetector barcodeDetector,
                         ContourDetector detector,
                         byte[] image,
                         int previewWidth,
                         int previewHeight,
                         CameraDetectorListener listener,
                         boolean detectBarcode) {
        this.detectBarcode = detectBarcode;
        this.barcodeDetector = barcodeDetector;
        this.detector = detector;
        this.image = image;
        this.previewWidth = previewWidth;
        this.previewHeight = previewHeight;

        this.listener = listener;
    }

    @Override
    public void run() {
        final DetectionResult detectionResult = detector.detect(image, previewWidth, previewHeight);
        final List<PointF> polygon = detector.getPolygonF();
        DebugLog.d("CAMERA_PREVIEW", detectionResult.name());
        DebugLog.d("CAMERA_PREVIEW", TextUtils.join(";", polygon));

        if (detectBarcode) {
            detectBarcode(image, previewWidth, previewHeight);
        }
        notifyCameraDetectorListener(detectionResult, polygon, null);
    }

    /**
     * Decode the data within the viewfinder rectangle, and time how long it took. For efficiency,
     * reuse the same reader objects from one decode to the next.
     *
     * @param data   The YUV preview frame.
     * @param width  The width of the preview frame.
     * @param height The height of the preview frame.
     */
    private void detectBarcode(byte[] data, int width, int height) {
        PlanarYUVLuminanceSource source = new PlanarYUVLuminanceSource(data, width, height, 0, 0,
                width, height, false);
        BinaryBitmap bitmap = new BinaryBitmap(new HybridBinarizer(source));
        Result rawResult = barcodeDetector.decodeWithState(bitmap);

        if (rawResult != null) {
            // Don't log the barcode contents for security.
            notifyCameraDetectorListener(DetectionResult.OK_BARCODE, null, rawResult.getText());
        }
    }

    private void notifyCameraDetectorListener(final DetectionResult detectionResult, final List<PointF> polygon, final String content) {
        new Handler(Looper.getMainLooper()).post(new Runnable() {
            @Override
            public void run() {
                switch (detectionResult) {
                    case OK:
                        listener.onDetectionOK(polygon);
                        return;
                    case OK_BUT_BAD_ANGLES:
                    case OK_BUT_BAD_ASPECT_RATIO:
                    case OK_BUT_TOO_SMALL:
                        listener.onDetectionWithError(detectionResult, polygon);
                        return;
                    case OK_BARCODE:
                        listener.onBarcodeDetectionOK(content);
                        return;
                    case ERROR_NOTHING_DETECTED:
                    case ERROR_TOO_NOISY:
                    case ERROR_TOO_DARK:
                        listener.onDetectionFailed(detectionResult);
                        return;
                }
            }
        });
    }
}
